#include "wam.h"
#include "program.h"
#include "useful.h"
#include <memory>
#include "unification.h"

namespace lp {
	// Global Stack
	std::array<term,HEAP_SIZE> heap;
	int heap_i = 0; // H
	int heap_back = 0; // HB
	int env_i = 0; // E
	int env_back = 0; // B
	int env_back0 = 0; // B0
	int stack_bottom = 0;
	int num_of_args = 0;
	// Read/Write mode
	wam_mode_t wam_mode;
	term* s_ptr = &heap[0]; // pointer to hs for reading arguments
	// Local Stack
	std::array<term,STACK_SIZE> stack;
	// Variable registers
	std::array<term*,VAR_SIZE> var_regs;
	// Trail
	std::array<term*,TRAIL_SIZE> trail;
	int tindex = 0; // TR
	// Runtime info
	int qlevel; // query level
	int qlevel2;
	// Code pointer
	code_iterator cap;
	code_iterator cap2;
	// Current Working Directory
	boost::filesystem::path CURRENT_PATH;

	// Variable mapping X -> heap/stack address
	std::map<id_type,term*> var_map;
	// Variable mapping position -> heap/stack address
	std::vector<term*> arg_map;

	// Substitutions in answers
	//answer_type answers;

	// Set static counter
	unsigned long long vinfo::counter = 0;

	std::map<id_type,vinfo> make_variable_table(
		const Functor& f, 
		bool collect, 
		int& permc, 
		int& tempc,
		int& max_argc,
		int& cl_len,
		int& last_cut)
	{
		// Keep table of variable names and their first/last occurrence
		std::map<id_type,vinfo> vi;
		max_argc = 0;

		// Grab all variables
		if (f.head()) {
			int argk = 0;
			std::for_each(f.head()->arg_begin(),f.head()->arg_end(),[&](const Functor* a){
				++argk;
				if (a->is_variable()) {
					// Argument variable
					auto at = vi.find(a->id());
					if (at == vi.end()) {
						vi.insert(std::make_pair(a->id(), vinfo(a->id(),0,false,argk)));
					} else {
						++(at->second.count);
						at->second.head_upos = -1;
					}
				} else if (a->is_function()) {
					std::for_each(++a->begin(),a->end(),[&](const Functor& n){
						if (n.is_variable()) {
							// Nested variable
							auto at = vi.find(n.id());
							if (at == vi.end()) {
								vi.insert(std::make_pair(n.id(), vinfo(n.id(),0,true)));
							} else {
								++(at->second.count);
								at->second.head_upos = -1;
								at->second.nested = true;
							}
						}
					});
				}
			});
			if (argk > max_argc) max_argc = argk;
		}

		int li = 0;
		cl_len = 0;
		last_cut = 0;
		for (auto l = f.body_begin(); l != f.body_end(); ++l) {
			const Functor& lit = *l;
			if (lit.is_constant(true_id)) continue;
			++li; // only increase for valid body literals (not "true")
			if (lit.is_constant(cut_id)) {
				last_cut = li;
			} else if (lit.is_variable()) {
				// dynamic call: just make sure to register variable to avoid incorrect optimization
				auto at = vi.find(lit.id());
				if (at != vi.end()) {
					++at->second.count;
					at->second.last = li;
					at->second.body_upos = 0;
					//std::cerr << "make_variable_table, lit is variable\n";
				} else {
					auto p = vi.insert(std::make_pair(lit.id(), vinfo(lit.id(),li,false)));
					p.first->second.body_upos = 0;
				}
			}
			++cl_len; // increase number of clauses
			int argc = 0;
			std::for_each(lit.arg_begin(),lit.arg_end(),[&](const Functor* a){
				++argc;
				if (a->is_variable()) {
					auto at = vi.find(a->id());
					if (at == vi.end()) {
						// This is the first (and only) occurrence
						vi.insert(std::make_pair(a->id(), vinfo(a->id(),li,false)));
					} else {
						// Already has first occurrence, update last
						at->second.last = li;
						if (li == 1 && at->second.head_upos > 0 && at->second.count == 1) {
							assert( at->second.body_upos == -1 );
							at->second.body_upos = argc;
						} else {
							at->second.body_upos = -1;
						}
						++(at->second.count);
					}
				} else if (a->is_function()) {
					std::for_each(++a->begin(),a->end(),[&](const Functor& n){
						if (n.is_variable()) {
							auto at = vi.find(n.id());
							if (at == vi.end()) {
								// This is the first (and only) occurrence
								vi.insert(std::make_pair(n.id(), vinfo(n.id(),li,true)));
							} else {
								// Already has first occurrence, update last
								at->second.last = li;
								++(at->second.count);
								at->second.body_upos = -1;
								at->second.nested = true;
							}
						}
					});
				}
			});

			// update max argument size
			if (argc > max_argc) max_argc = argc; 
		}

		/* Determine which variables are temporary and permanent
		Definition: Temporary <=> 
		A. Occurs in only one body goal (head is part of first body goal), 
		AND at least one of the following:
		B1. first occurs in the head
		B2. first occurs in nested struct
		B3. first occurs in last goal
		Note that permanent variables need indices Y1,Y2,...
		*/
		assert(cl_len == std::count_if(f.body_begin(),f.body_end(),[&](const Functor& l){ return !l.is_constant(true_id); }));
		for (auto v = vi.begin(); v != vi.end(); ++v) {
			if (v->second.last <= 1 || v->second.first == cl_len /*|| v->second.first == v->second.last*/) {
				assert( v->second.last > 1 || v->second.first <= 1 );
				assert( v->second.first != cl_len || v->second.last == cl_len );
				// Note: if first appears in last (cl_len), then it ONLY appears in last
				// Temporary
				v->second.temporary = true;
			} else {
				// Permanent
				v->second.temporary = false;
			}
		}

		// Map variables
		std::set<vinfo*,cmp_temporary> vt; // temporary ordered according to when they appeared
		std::set<vinfo*,cmp_permanent> vp; // permanents ordered from last goal
		for (auto v = vi.begin(); v != vi.end(); ++v) {
			if (v->second.head_upos > 0 && (v->second.count == 2 && v->second.head_upos == v->second.body_upos || v->second.count == 1)) {
				assert(v->second.first == 0);
				// This variable occurs only once in head and at most once in body, at same argument position
				// It doesn't need to be indexed!
				v->second.index = -1; // does not have an index
			} else if (v->second.first == 0 && v->second.last == 0 && v->second.count > 1 && !v->second.nested) {
				v->second.index = -1; // no index (peephole optimization)
			} else if (v->second.head_upos > 0 && v->second.body_upos == 0 && v->second.last == 1 && v->second.count == 2 && !v->second.nested) {
				assert(v->second.first == 0);
				//std::cerr << "// p(X,Y) :- X, ...\n";
				v->second.index = -1; // dynamic call optimization
			} else if (v->second.temporary) {
				vt.insert(&v->second);
			} else {
				vp.insert(&v->second);
			}
		}
		// Set number of permanent variables
		permc = vp.size();
		// Set number of temporary variables
		tempc = vt.size();
		// Assign indices
		int idx = 0;
		for (auto p : vp) p->index = idx++;
		idx = 0;
		for (auto p : vt) p->index = idx++;
		//idx = (f->head() ? f->head()->arity() : 0) + 1; // leave argument registers alone
		//for (auto i = vt.begin(); i != vt.end(); ++i) {
		//	i->index = idx++;
		//}

		return vi;
	}

	std::vector<const Functor*> make_equations(const Functor& t, std::map<const Functor*,int>& vlink)
	{
		std::vector<const Functor*> eqn, veqn;
		// Build arguments
		int vi = 0;
		//std::cerr << "make equations: " << *t << "\n";
		for (auto a = t.arg_begin(); a != t.arg_end(); ++a) {
			//std::cerr << "  argument: " << *a << "\n";
			eqn.push_back(*a);
			for (auto n = ++((*a)->begin()); n != (*a)->end(); ++n) {
				if (n->is_function()) {
					//std::cerr << "  vlink: " << *n << " [" << (vi+1) << "]\n";
					veqn.push_back(&*n);
				}
			}
		}
		for (auto p : veqn) {
			vlink.insert( std::make_pair(p, eqn.size()) );
			eqn.push_back(p);
		}
		return eqn;
	}


	void make_wam_instructions(
		const Functor& cl,
		const std::set<id_type>& qvars,
		bool collect,
		clause* ci)
	{
		//std::cerr << "make_wam_instructions: " << cl << "\n";
		assert(!collect || !cl.head());
		//std::cerr << "is query? " << is_query << "\n";
		// Get variable types and order
		int permc; // get number of permanent variables
		int tmpc; // get number of temporary variables (note that permc + temc + optimized away vars = vtbl.size())
		int argc; // get max number of arguments
		int cl_len; // clause length
		int last_cut; // position of last cut
		std::map<id_type,vinfo> vtbl = make_variable_table(cl,collect,permc,tmpc,argc,cl_len,last_cut);
		//std::cerr << "argc: " << argc << "\n";
		int perm_left = permc; // permanent left
		int prev_perm_left = perm_left; // previous permanent_left, used for delayed trimming
		assert(argc >= 0 && tmpc >= 0 && perm_left >= 0 && permc >= 0);
		// Keep track of which variables are on the heap
		std::set<id_type> global;
		// Store all unsafe values (for PUT_UNSAFE_VALY)
		std::set<id_type> unsafe;
		// Keep track of variables we've seen
		std::set<id_type> vseen;
		// Keep track of peephole optimizations
		std::map<id_type,int> peephole;
		// Keep track of variable links (due to flattening)
		std::map<const Functor*,int> vlink;
		// Delayed trimming
		std::set<id_type> delay_trimming;
		// Keep track of registered variables
		std::set<id_type> regx,regy;

		// Allocate: we only need an environment if we have more than 1 body literal
		// Cut: check if there is a cut while scanning
		bool alloc_env = false;
		// Allocate environment?
		/* Note: environment stores cap2 pointer (as well as permanent variables).
			cap2 ptr is needed when we have an external CALL (i.e. non-builtin) as we may call PROCEED (which ruins cap2)
		*/
		if (last_cut >= 2) ++permc; // add one permanent var if we use a deep cut
		if (cl_len > 1) {
			// Note: even if permc == 0, we need an environment for cap2
			ci->push_back(instruction(instruction::ALLOC));
			alloc_env = true;
		}
		if (last_cut >= 2) {
			// we have a deep cut, use get_level
			ci->push_back(instruction(instruction::GET_LVL,permc));
		}

		// Head
		if (cl.head()) {
			// Flatten head
			auto eqn = make_equations(*cl.head(),vlink);

			//std::cerr << "Head: " << *cl.head() << "\n";
			//std::cerr << "Eqn: " << eqn.size() << "\n";
			//std::cerr << "Vtbl: " << vtbl.size() << "\n";
			//std::for_each(eqn.begin(),eqn.end(),[&](const term* p){
			//	std::cerr << "  eq: " << *p << "\n";
			//});

			// For each equation in Head
			auto head_nested = [&](const Functor& t, const std::map<const Functor*,int>& vlink) -> void 
			{
				if (t.is_variable()) {
					auto at = vtbl.find(t.id());
					const vinfo& vi = at->second;
					assert(vi.index >= 0);
					const bool is_new = (vseen.find(t.id()) == vseen.end());
					if (vi.count == 1) {
						assert(is_new);
						ci->push_back(instruction(instruction::UNI_VOID,1));
					} else if (vi.temporary) { // Xi
						if (is_new) {
							ci->push_back(instruction(instruction::UNI_VAR, argc + vi.index));
							global.insert(t.id());
							vseen.insert(t.id());
						} else {
							if (global.find(t.id()) == global.end()) {
								// NOT globalized
								ci->push_back(instruction(instruction::UNI_LOCAL_VAL, argc + vi.index));
								global.insert(t.id());
							} else {
								ci->push_back(instruction(instruction::UNI_VAL, argc + vi.index));
							}
						}
					} else { // Yi
						if (is_new) {
							ci->push_back(instruction(instruction::UNI_VARY, vi.index));
							global.insert(t.id());
							vseen.insert(t.id());
						} else {
							if (global.find(t.id()) == global.end()) {
								// NOT globalized
								ci->push_back(instruction(instruction::UNI_LOCAL_VALY, vi.index));
								global.insert(t.id());
							} else {
								ci->push_back(instruction(instruction::UNI_VALY, vi.index));
							}
						}
					}
				} else if (t.is_int()) {
					ci->push_back(instruction(instruction::UNI_INT, t.to_int()));
				} else if (t.is_double()) {
					ci->push_back(instruction(instruction::UNI_FLT, t.to_double()));
				} else if (t.is_constant()) {
					ci->push_back(instruction(instruction::UNI_CON, t.id()));
				} else {
					// Functor nested => use variable linking
					auto at = vlink.find(&t);
					assert(at != vlink.end());
					ci->push_back(instruction(instruction::UNI_VAR, argc + tmpc + at->second));
				}
			};
			auto head_argument = [&](const Functor& t, int argi, const std::map<const Functor*,int>& vlink) -> void 
			{
				if (t.is_variable()) {
					auto at = vtbl.find(t.id());
					const vinfo& vi = at->second;
					const bool is_new = (vseen.find(t.id()) == vseen.end());
					if (vi.index < 0) {
						if (vi.first == 0 && vi.last == 0 && vi.count > 1 && !vi.nested) {
							// Peephole Optimization
							if (is_new) {
								// only register instance
								peephole.insert(std::make_pair(t.id(),argi));
							} else {
								auto at = peephole.find(t.id());
								ci->push_back(instruction(instruction::GET_VAL, at->second, argi));
							}
						} else {
							// Optimize away this variable
							//assert(vi.first == 0 && vi.last == 0 && vi.count == 1 && !vi.nested
							//	|| vi.first == 0 && vi.last == 1 && vi.count == 2 && !vi.nested);
							//assert(vi.head_upos >= 0 && (vi.head_upos == vi.body_upos || vi.count == 1));
						}
					} else if (vi.temporary) { // Xi
						//std::cerr << "GET_VAR, argc: " << argc << ", vi.index: " << vi.index << ", " << argi << "\n";
						ci->push_back(is_new ? instruction(instruction::GET_VAR, argc + vi.index, argi)
							: instruction(instruction::GET_VAL, argc + vi.index, argi));
					} else { // Yi
						ci->push_back((is_new ? instruction(instruction::GET_VARY, vi.index, argi)
							: instruction(instruction::GET_VALY, vi.index, argi)));
					}
					if (is_new) vseen.insert(t.id());
				} else if (t.is_int()) {
					ci->push_back(instruction(instruction::GET_INT, t.to_int(), argi));
				} else if (t.is_double()) {
					ci->push_back(instruction(instruction::GET_FLT, t.to_double(), argi));
				} else if (t.is_constant()) {
					ci->push_back(instruction(instruction::GET_CON, t.id(), argi));
				} else {
					assert( t.is_function() );
					if (t.is_pair()) {
						ci->push_back(instruction(instruction::GET_LIS, argi));
						head_nested(*t.arg_first(),vlink);
						head_nested(*t.arg_last(),vlink);
					} else {
						ci->push_back(instruction(instruction::GET_STR, t.id(), t.arity(), argi));
						// Add all arguments
						for (auto a = t.arg_begin(); a != t.arg_end(); ++a) {
							head_nested(**a,vlink);
						}
					}
				}
			};

			decltype(cl.head()->arity()) k = 0;
			// The first cl.head()->arity() equations are arguments
			for (auto e = eqn.begin(); e != eqn.end(); ++e, ++k) {
				//std::cerr << "  eqn: " << *e << "\n";
				if (k < cl.head()->arity()) {
					head_argument(**e,k,vlink);
				} else {
					// These are variable linked structures
					head_argument(**e, argc + tmpc + k, vlink);
				}
			} // for each equation
		} // end of head

		// Body
		int litc = 0;
		bool used_execute = false;

		auto body_nested = [&](const Functor& t, const std::map<const Functor*,int>& vlink) -> void 
		{
			if (t.is_variable()) {
				const bool is_new = (vseen.find(t.id()) == vseen.end());
				assert( vlink.find(&t) == vlink.end() );
				// Temporary or Permanent Variable?
				auto at = vtbl.find(t.id());
				assert(at != vtbl.end());
				const vinfo& vi = at->second;
				assert(vi.index >= 0);
				if (vi.count == 1) {
					assert( is_new );
					ci->push_back(instruction(instruction::SET_VOID,1));
					if (collect) {
						//auto at = qvars.find(t.id());
						//if (at != qvars.end()) {
							const data& dat = lp::fmap.get_data(t.id());
							assert( dat.is_atom() );
							ci->push_back(instruction(instruction::REG_HVAR,t.id()));
							regx.insert(t.id());
						//}
					}
					vseen.insert(t.id()); // just in case we need it elsewhere
				} else if (vi.temporary) { // Xi
					if (is_new) {
						ci->push_back(instruction(instruction::SET_VAR, argc + vi.index));
						if (collect) {
							// Register heap variable
							//auto at = qvars.find(t.id());
							if (/*at != qvars.end() &&*/ regx.find(t.id()) == regx.end()) {
								ci->push_back(instruction(instruction::REG_HVAR,t.id()));
								regx.insert(t.id());
							}
						}
						global.insert(t.id());
						vseen.insert(t.id());
					} else {
						if (global.find(t.id()) == global.end()) {
							// NOT globalized
							ci->push_back(instruction(instruction::SET_LOCAL_VAL, argc + vi.index));
							global.insert(t.id());
						} else {
							ci->push_back(instruction(instruction::SET_VAL, argc + vi.index));
						}
					}
				} else { // Yi
					if (is_new) {
						ci->push_back(instruction(instruction::SET_VARY,vi.index));
						if (collect) {
							// Register stack variable
							//auto at = qvars.find(t.id());
							if (/*at != qvars.end() &&*/ regy.find(t.id()) == regy.end()) {
								ci->push_back(instruction(instruction::REG_SVAR,t.id(),vi.index));
								regy.insert(t.id());
							}
						}
						global.insert(t.id());
						vseen.insert(t.id());
					} else {
						if (global.find(t.id()) == global.end()) {
							// NOT globalized
							ci->push_back(instruction(instruction::SET_LOCAL_VALY,vi.index));
							global.insert(t.id());
						} else {
							ci->push_back(instruction(instruction::SET_VALY,vi.index));
						}
					}
				}
			} else if (t.is_int()) {
				ci->push_back(instruction(instruction::SET_INT, t.to_int()));
			} else if (t.is_double()) {
				ci->push_back(instruction(instruction::SET_FLT, t.to_double()));
			} else if (t.is_constant()) {
				ci->push_back(instruction(instruction::SET_CON, t.id()));
			} else {
				// Functor nested => use variable linking
				auto at = vlink.find(&t);
				assert(at != vlink.end());
				ci->push_back(instruction(instruction::SET_VAL, argc + tmpc + at->second));
			}
		};

		auto body_argument = [&](const Functor& t, int argi, const std::map<const Functor*,int>& vlink) -> void 
		{
			//std::cerr << "  Body argument: " << t << "\n";
			if (t.is_variable()) {
				auto at = vtbl.find(t.id());
				const vinfo& vi = at->second;
				const bool is_new = (vseen.find(t.id()) == vseen.end());
				if (vi.index < 0) {
					// Optimize away this variable
					assert(vi.first == 0 && vi.last == 1 && vi.count == 2 && !vi.nested);
					assert(vi.head_upos >= 0 && (vi.head_upos == vi.body_upos || vi.count == 1));
				} else {
					// If variable only occurs in this goal, we need to delay trimming
					if (!vi.temporary && vi.first == vi.last && vi.first < cl_len) {
						// Delay trimming (and do NOT use PUT_UNSAFE)
						//std::cerr << "DELAY_TRIMMING\n";
						delay_trimming.insert(t.id());
					}
					if (vi.temporary) { // Xi
						if (is_new) {
							ci->push_back(instruction(instruction::PUT_VAR, argc + vi.index, argi));
							if (collect) {
								// Register heap variable?
								//auto at = qvars.find(t.id());
								//std::cerr << "var2id size: " << var2id.size() << "\n";
								//std::cerr << "variable found? " << (at != var2id.end()) << " (" << t << ")\n";
								//if (at != qvars.end()) {
									ci->push_back(instruction(instruction::REG_HVAR, t.id()));
									regx.insert(t.id());
								//}
							}
							vseen.insert(t.id());
							global.insert(t.id());
						} else {
							ci->push_back(instruction(instruction::PUT_VAL, argc + vi.index, argi));
						}
					} else { // Yi
						if (is_new) {
							ci->push_back(instruction(instruction::PUT_VARY, vi.index, argi));
							if (collect) {
								// Register stack variable?
								//auto at = qvars.find(t.id());
								//if (at != qvars.end()) {
									ci->push_back(instruction(instruction::REG_SVAR, t.id(), vi.index));
									regy.insert(t.id());
								//}
							}
							unsafe.insert(t.id());
							vseen.insert(t.id());
							// note: does NOT globalize
						} else {
							// Put Unsafe Value?
							// Note: put_unsafe is not used when delayed trimming is active, as it would undo the optimization
							decltype(unsafe.find(t.id())) at2;
							const bool use_unsafe = 
								(vi.last == litc && (at2 = unsafe.find(t.id())) != unsafe.end() && delay_trimming.find(t.id()) == delay_trimming.end());
							if (use_unsafe) {
								// unsafe value
								ci->push_back(instruction(instruction::PUT_UNSAFE_VALY, vi.index, argi));
								unsafe.erase(at2);
							} else {
								ci->push_back(instruction(instruction::PUT_VALY, vi.index, argi));
							}
						}
					}
					// Register argument?
					if (collect && argi <= argc) ci->push_back(instruction(instruction::REG_ARG, argi));
				} // if variable wasn't optimized away
			} else if (t.is_int()) {
				//std::cerr << "  body arg is int\n";
				ci->push_back(instruction(instruction::PUT_INT, t.to_int(), argi));
				if (collect && argi <= argc) ci->push_back(instruction(instruction::REG_ARG, argi));
				//std::cerr << "  body arg int done\n";
			} else if (t.is_double()) {
				ci->push_back(instruction(instruction::PUT_FLT, t.to_double(), argi));
				if (collect && argi <= argc) ci->push_back(instruction(instruction::REG_ARG, argi));
			} else if (t.is_constant()) {
				ci->push_back(instruction(instruction::PUT_CON, t.id(), argi));
				if (collect && argi <= argc) ci->push_back(instruction(instruction::REG_ARG, argi));
			} else {
				assert( t.is_function() );
				if (t.is_pair()) {
					ci->push_back(instruction(instruction::PUT_LIS, argi));
					if (collect && argi <= argc) ci->push_back(instruction(instruction::REG_ARG, argi));
					// Add all arguments
					body_nested(*t.arg_first(),vlink);
					body_nested(*t.arg_last(),vlink);
				} else {
					ci->push_back(instruction(instruction::PUT_STR, t.id(), t.arity(), argi));
					if (collect && argi <= argc) ci->push_back(instruction(instruction::REG_ARG, argi));
					// Add all arguments
					for (auto a = t.arg_begin(); a != t.arg_end(); ++a) {
						body_nested(**a,vlink);
					}
				}
			}
		};

		for (auto lit = cl.body_begin(); lit != cl.body_end(); ++lit) {
			if (lit->is_constant(true_id)) continue;
			++litc; // start at 1
			//std::cerr << "make_wam, body literal: " << **lit << "\n";
			std::vector<const Functor*> eqn;
			delay_trimming.clear();

			if (last_cut == litc) {
				assert( lit->is_constant(lp::cut_id));
				if (litc == 1) {
					// neck cut
					ci->push_back(instruction(instruction::NECK_CUT));
				} else {
					ci->push_back(instruction(instruction::CUT, permc));
				}
				// Note: still proceed to add PROCEED / DEALLOCATE
			} else {
				// Make equations consisting of arguments
				eqn = make_equations(*lit,vlink);
				//std::cerr << "equations: " << eqn.size() << "\n";
			}

			// For each equation
			/* NOTE:
			eqn.size() is NOT number of arguments, as we may also have flattened expressions 
			The number of arguments (for this literal) is given by lit->arity()
			The number of arguments in total (max) is given by argc
			*/
			// Note: for queries, we add in reverse order
			decltype(eqn.begin()) e;
			decltype(lit->arity()) k;
			e = (eqn.empty() ? eqn.end() : --eqn.end());
			k = (eqn.empty() ? 0 : eqn.size()-1);
			// For each equation
			for ( ; e != eqn.end(); ) {
				//std::cerr << "  eqn: "; print_infix(std::cerr,*e); cerr << "\n";
				if (k < lit->arity()) {
					//std::cerr << "body argument: "; print_infix(cerr,*e); cerr << " (" << k << ")\n";
					body_argument(**e,k,vlink);
				} else {
					// These are variable linked structures
					//std::cerr << "body nested: "; print_infix(cerr,*e); cerr << "\n";
					body_argument(**e, argc + tmpc + k, vlink);
				}
				//std::cerr << "done with this equation\n";
				// Move to next equation
				if (e == eqn.begin()) break;
				--e;
				--k;
			} // for each equation

			// Update number of remaining permanent variables needed
			prev_perm_left = perm_left;
			perm_left -= std::count_if(vtbl.begin(),vtbl.end(),[&](const std::map<id_type,vinfo>::value_type& p){
				return !p.second.temporary && p.second.last == litc;
			});
			// Dealloc if needed
			if (litc == cl_len && alloc_env) {
				if (!regy.empty()) {
					ci->push_back(instruction(instruction::GLOBALIZE_SVARS));
				}
				ci->push_back(instruction(instruction::DEALLOC));
			}
			// Insert CALL / EXECUTE / builtin instruction
			if ( lit->is_variable() || lit->is_function(call_id,1) ) {
				const id_type var = (lit->is_variable() ? lit->id() : lit->arg_first()->id());
				auto at = vtbl.find(var);
				if (at == vtbl.end()) {
					// Variable unbound
					ci->push_back(instruction(instruction::FAIL));
				} else {
					const vinfo& vi = at->second;
					// Variable as body literal => dynamic_call
					//std::cerr << "variable as body literal\n";
					if (litc == cl_len) {
						//std::cerr << "last literal\n";
						assert( delay_trimming.empty() );
						if (!cl.head()) {
							//std::cerr << "cl has no head, use DYN_CALL\n";
							// Queries/Negative examples never use execute(), end with PROMPT
							if (vi.index < 0) {
								// Dynamic call optimization
								assert(vi.head_upos > 0);
								ci->push_back(instruction(instruction::DYN_CALL, vi.head_upos-1, perm_left + (litc < last_cut)));
							} else if (vi.temporary) {
								ci->push_back(instruction(instruction::DYN_CALL, argc + vi.index, perm_left + (litc < last_cut)));
							} else {
								ci->push_back(instruction(instruction::DYN_STACK_CALL, vi.index, perm_left + 1 + (litc < last_cut)));
							}
							// Note: DYN_STACK_CALL must delay its permanent variable (alternative: globalize it)
						} else {
							//std::cerr << "cl has head, use DYN_EXEC; argc: " << argc << ", vi.index: " << vi.index << "\n";
							if (vi.index < 0) ci->push_back(instruction(instruction::DYN_EXEC, vi.head_upos-1));
							else if (vi.temporary) ci->push_back(instruction(instruction::DYN_EXEC, argc + vi.index));
							else ci->push_back(instruction(instruction::DYN_STACK_EXEC, argc + vi.index));
							used_execute = true;
						}
					} else {
						//std::cerr << "not last literal\n";
						if (delay_trimming.empty()) {
							if (vi.temporary) ci->push_back(instruction(instruction::DYN_CALL, argc + vi.index, perm_left + (litc < last_cut)));
							else ci->push_back(instruction(instruction::DYN_STACK_CALL, vi.index, perm_left + 1 + (litc < last_cut)));
						} else {
							// Delay trimming: at least one iteration
							assert(!vi.temporary);
							ci->push_back(instruction(instruction::DYN_STACK_CALL, vi.index, prev_perm_left + 1 + (litc < last_cut)));
						}
					}
				}
			} else if ( !lit->is_constant(lp::cut_id) ) {
				const sign_t p = lit->signature();
				auto at = builtin.find(p);
				//std::cerr << "CHECKING IF " << lp::fmap.get_data(lit->get_fun()) << "/" << lit->arity() << " is BUILTIN\n";
				if (at == builtin.end()) {
					// Not builtin, use CALL/EXECUTE
					if (litc == cl_len) {
						if (!cl.head()) {
							// Queries/Negative examples never use execute(), end with PROMPT
							ci->push_back(instruction(instruction::CALL,p.first,p.second,perm_left + (litc < last_cut)));
						} else {
							ci->push_back(instruction(instruction::EXEC,p.first,p.second));
							used_execute = true;
						}
					} else {
						if (delay_trimming.empty()) {
							ci->push_back(instruction(instruction::CALL,p.first,p.second,perm_left + (litc < last_cut)));
						} else {
							// Delay trimming: keep all permanent variables in this
							//std::cerr << "CALL with DELAY_TRIM: " << prev_perm_left + (litc < last_cut) + 1 << "\n";
							ci->push_back(instruction(instruction::CALL,p.first,p.second,prev_perm_left + (litc < last_cut)));
						}
					}
				} else {
					// Builtin
					ci->push_back(instruction( at->second, 0, 1, 2 ));
				}
			} // if not cut operator
		} // for each body literal

		if (!cl.head()) {
			// Queries/negative examples end with PROMPT
			ci->push_back(instruction(instruction::PROMPT));
		} else {
			// Facts/Rules end with EXECUTE or PROCEED
			if (!used_execute) {
				ci->push_back(instruction(instruction::PROCEED));
			}
		}

		//return ci;
	}



	int wam_on_init()
	{
		/* Insert code for '\+'/1, equivalent to:
		not(Goal) :- Goal, !, fail.
		not(Goal).
		*/
		//instruction* cptr = &code_area[BUILTIN_UNIFY];
		//*cptr++ = instruction(instruction::GET_VAR,var_regs[3],var_regs[1]);
		//*cptr++ = instruction(instruction::GET_VAL,var_regs[3],var_regs[1]);
		//*cptr++ = instruction(instruction::TRY_ME_ELSE, (cptr - &code_area[0]) + 6);
		//*cptr++ = instruction(instruction::ALLOC);
		//*cptr++ = instruction(instruction::DYNCALL,&hs[VAR_BEGIN]);
		//*cptr++ = instruction(instruction::CUT);
		//*cptr++ = instruction(instruction::FAIL);
		//*cptr++ = instruction(instruction::DEALLOC);
		//*cptr++ = instruction(instruction::TRUST_ME);
		////*cptr++ = instruction(instruction::TRUE);
		//*cptr++ = instruction(instruction::PROCEED);
		return 1;
	}


	inline term* make_op(id_type id, std::vector<term>& args)
	{
		if (args.empty()) {
			return new term(term::CON,id);
		} else {
			term* s = new term[args.size()+1];
			s[0] = term::make_function(id,args.size());
			for (unsigned k = 1; k <= args.size(); ++k) {
				s[k] = std::move(args[k]);
			}
			return s;
		}
	}

	sign_t get_sign(const instruction*& i)
	{
		sign_t s(0,0);
		bool finished = false;
		for ( ; !finished; ++i) {
			//std::cerr << "get_sign, i: " << *i << "\n";
			switch (i->opcode) {
			case instruction::PUT_CON:
			case instruction::PUT_INT:
			case instruction::PUT_FLT:
			case instruction::PUT_STR:
			case instruction::PUT_LIS:
			case instruction::PUT_VAR:
			case instruction::PUT_VARY:
			case instruction::PUT_VAL:
			case instruction::PUT_VALY:
			case instruction::PUT_UNSAFE_VALY:
				break;
			case instruction::CALL: 
			case instruction::EXEC: 
				s.first = i->a1;
				s.second = i->a2;
				//std::cerr << "s.second: " << s.second << "\n";
				finished = true;
				break;
			case instruction::DYN_CALL: 
			case instruction::DYN_EXEC:
			case instruction::DYN_STACK_CALL: 
			case instruction::DYN_STACK_EXEC:
				s.first = call_id;
				s.second = 1;
				finished = true;
				break;
			case instruction::NECK_CUT:
			case instruction::CUT:
				s.first = cut_id;
				s.second = 0;
				finished = true;
				break;
			case instruction::PROCEED:
				assert( s.first == 0 && s.second == 0 );
				// fall through
			case instruction::PROMPT:
				// Nothing more to build
				finished = true;
				break;
			case instruction::DEALLOC:
			case instruction::GLOBALIZE_SVARS:
			case instruction::NOOP:
			case instruction::TRY_ME_ELSE:
			case instruction::RETRY_ME_ELSE:
			case instruction::TRUST_ME:
				break;
			default:
				{
					auto at = find_if(builtin.begin(),builtin.end(),
						[&](const builtin_lookup_type::value_type& p){
							return p.second == i->opcode;
					});
					if (at != builtin.end()) {
						s = at->first;
						finished = true;
					} else {
						//std::cerr << "Decompile_bl, ignoring " << *j << "\n";
					}
				}
			} // switch
		} // for
		return s;
	}


	bool is_ground(const sign_t& s, const instruction* i)
	{
		// Handle head
		bool finished = false;
		for ( ; !finished ; ++i) {
			switch (i->opcode) {
				case instruction::UNI_VAR:
				case instruction::UNI_VAL:
				case instruction::UNI_LOCAL_VAL:
					// Could be variable link
					if (i->a1 < s.second) {
						// Argument is variable => not ground
						//std::cerr << "is_ground, not link: " << i->a2 << " > " << s.second << "\n";
						//std::cerr << "  instruction: " << *i << "\n";
						return false;
					} // else: variable link => ok
					break;
				case instruction::UNI_VARY:
				case instruction::UNI_VALY:
				case instruction::UNI_LOCAL_VALY:
				case instruction::GET_VAR:
				case instruction::GET_VAL:
				case instruction::GET_VARY:
				case instruction::GET_VALY:
					//std::cerr << "is_ground, variable: " << *i << "\n";
					return false;
				case instruction::PROCEED:
					return true; // no body => done
				case instruction::NOOP:
				case instruction::TRY_ME_ELSE:
				case instruction::RETRY_ME_ELSE:
				case instruction::TRUST_ME:
				case instruction::GET_LVL:
				case instruction::ALLOC:
					// Ignore
					break;
				default:
					finished = true;
					break;
				}
		} // for
		--i; // from for loop
		// Handle body
		return is_ground(i);
	}

	bool is_ground(const instruction* i)
	{
		for (;;) {
			auto j = i;
			const sign_t s = get_sign(j);
			//std::cerr << "is_ground, get_sign: " << fmap.get_data(s.first) << "/" << s.second << "\n";
			if (s.first == 0) break; // finished
			for ( ; i != j; ++i) {
				//std::cerr << "is_ground, i: " << *i << "\n";
				switch (i->opcode) {
				case instruction::SET_VAR:
				case instruction::SET_VAL:
				case instruction::SET_LOCAL_VAL:
				case instruction::UNI_VAR:
				case instruction::UNI_VAL:
				case instruction::UNI_LOCAL_VAL:
					// Could be variable link
					if (i->a1 < s.second) {
						//std::cerr << "not ground: i->a2: " << i->a2 << " < " << s.second << "\n";
						// Argument is variable => not ground
						return false;
					}
					// else: variable link => ok
					break;
				case instruction::SET_VARY:
				case instruction::SET_VALY:
				case instruction::SET_LOCAL_VALY:
				case instruction::UNI_VARY:
				case instruction::UNI_VALY:
				case instruction::UNI_LOCAL_VALY:
				case instruction::GET_VAR:
				case instruction::GET_VAL:
				case instruction::GET_VARY:
				case instruction::GET_VALY:
				case instruction::PUT_VAR:
				case instruction::PUT_VAL:
				case instruction::PUT_VARY:
				case instruction::PUT_VALY:
				case instruction::PUT_UNSAFE_VALY:
					return false;
				default:
					// Ignore
					break;
				}
			} // for each instruction

			const instruction& pi = *std::prev(i);
			assert( pi.opcode != instruction::PROCEED ); // proceed is handled by giving s(0,0)
			if (pi.opcode == instruction::EXEC) {
				break;
			}
		} // while there are body literals
		return true;
	}

	Functor decompile_bl(
		const instruction*& i,
		std::map<int,id_type>& x2v,
		std::map<int,id_type>& y2v,
		const Functor* head,
		Subst& subs)
	{
		// Scan ahead to find number of arguments
		auto j = i;
		const sign_t s = get_sign(j);

		if (s.first == 0) {
			// No body literal
			return Functor();
		} else if (s.second == 0) {
			i = j; // update code pointer
			//std::cerr << "Decompile_bl, signature is: " << fmap.get_data(s.first) << "/0\n";
			return Functor(new Functor(s.first));
		}

	    // std::cerr << "Decompile_bl, signature is: " << fmap.get_data(s.first) << "/" << s.second << "\n";

		std::vector<Functor> bl(s.second,Functor());
		// Keep track of which variables were uninstantiated (if it's the first body literal)

		// map variable registers to allocated memory
		//std::map<int,term*>::iterator at;
		bool finished = false;

		for ( ; !finished; ++i) {
		    // std::cerr << "Decompile body literals, instruction: " << *i << "\n";
			switch (i->opcode) {
			case instruction::PUT_CON: bl[i->a2] = Functor(i->a1); break;
			case instruction::PUT_INT: bl[i->a2] = Functor::make_int(i->i1); break;
			case instruction::PUT_FLT: bl[i->a2] = Functor::make_double(i->d1); break;
			case instruction::PUT_VAR:
				{
					auto newvar = Functor::unique_index();
					bl[i->a2] = Functor(newvar);
					x2v.insert(std::make_pair(i->a1,newvar));
				}
				break;
			case instruction::PUT_VARY:
				{
					auto newvar = Functor::unique_index();
					bl[i->a2] = Functor(newvar);
					y2v.insert(std::make_pair(i->a1,newvar));
				}
				break;
			case instruction::PUT_VAL:
				bl[i->a2] = x2v.find(i->a1)->second;
				break;
			case instruction::PUT_VALY:
			case instruction::PUT_UNSAFE_VALY:
				bl[i->a2] = Functor(y2v.find(i->a1)->second);
				break;
			case instruction::PUT_STR:
			case instruction::PUT_LIS:
				{
					int reg;
					sign_t ss;
					if (i->opcode == instruction::PUT_STR) {
						ss = sign_t(i->a1,i->a2);
						reg = i->a3;
					} else {
						ss = sign_t(pair_id,2);
						reg = i->a1;
					}
					++i;
					Functor f(ss.first);
					f.arg_reserve(ss.second);
					for (int n = 0; n < ss.second; ++n,++i) {
					    // std::cerr << "Decompile bl, nested instruction: " << *i << "\n";
						switch (i->opcode) {
						case instruction::SET_CON: f.steal(new Functor(i->a1)); break;
						case instruction::SET_INT: f.steal(new Functor(Functor::make_int(i->i1))); break;
						case instruction::SET_FLT: f.steal(new Functor(Functor::make_double(i->d1))); break;
						case instruction::SET_VAR:
							{
								auto newvar = Functor::unique_index();
								x2v.insert(std::make_pair(i->a1,newvar));
								f.steal(new Functor(newvar));
							}
							break;
						case instruction::SET_VARY:
							{
								auto newvar = Functor::unique_index();
								y2v.insert(std::make_pair(i->a1,newvar));
								f.steal(new Functor(newvar));
							}
							break;
						case instruction::SET_VAL:
						case instruction::SET_LOCAL_VAL:
							f.steal(new Functor(x2v.find(i->a1)->second));
							break;
						case instruction::SET_VALY:
						case instruction::SET_LOCAL_VALY:
							f.steal(new Functor(y2v.find(i->a1)->second));
							break;
						case instruction::SET_VOID:
							f.steal(new Functor(Functor::unique_index()));
							break;
						// just ignore:
						case instruction::REG_ARG:
							--n;
							break;
						default:
							std::cerr << "ERROR: expected SET instruction, read: " << *i << "\n";
							assert(false);
							exit(1);
						}
					}
					--i;
					if (reg < s.second) {
						// argument
						//f.subst(subs);
						//std::cerr << "Argument: " << f << "\n";
						bl[reg] = std::move(f);
					} else {
						// link
						assert(x2v.find(reg) == x2v.end());
						const auto newvar = Functor::unique_index();
						x2v.insert(std::make_pair(reg,newvar));
						assert(!subs.get(newvar));
						subs.steal(newvar,new Functor(f));
					}
				}
				break;
			case instruction::CALL:
			case instruction::EXEC:
			case instruction::PROCEED:
				finished = true;
				break;
			case instruction::DYN_CALL:
			case instruction::DYN_EXEC:
				{
					auto at = x2v.find(i->a1);
					if (at != x2v.end()) {
						bl[0] = Functor(at->second);
					} else {
						// Dynamic call optimization used
						bl[0] = *head->arg(i->a1);
					}
					finished = true;
				}
				break;
			case instruction::DYN_STACK_CALL:
			case instruction::DYN_STACK_EXEC:
				{
					auto at = y2v.find(i->a1);
					if (at != y2v.end()) {
						bl[0] = Functor(at->second);
					}
					finished = true;
				}
				break;
			case instruction::DEALLOC:
			case instruction::GLOBALIZE_SVARS:
			case instruction::NOOP: 
			case instruction::TRY_ME_ELSE:
			case instruction::RETRY_ME_ELSE:
			case instruction::TRUST_ME:
				break;
			default:
				{
					auto at = find_if(builtin.begin(),builtin.end(),
						[&](const builtin_lookup_type::value_type& p){
							return p.second == i->opcode;
					});
					if (at != builtin.end()) {
						assert( s.first == at->first.first );
						assert( s.second == at->first.second );
						finished = true;
					} else {
						//std::cerr << "Decompile, ignoring: " << *i << "\n";
					}
				}
			}
		}

		Functor f(new Functor(s.first));
		int k = 0;
		for (auto&& a : bl) {
			if (a.is_nil()) {
				if (head) { 
					// optimization: copy from head
					f.steal(head->arg(k)->copy());
				} else {
					f.steal(new Functor(Functor::unique_index()));
				}
			} else {
				f.steal(new Functor(std::move(a)));
			}
			++k;
		}
		//f.subst(subs);
		return f;
	}

	// Make term from instructions
	Functor decompile_head(
		const sign_t& s, 
		const instruction*& i,
		std::map<int,id_type>& x2v,
		std::map<int,id_type>& y2v,
		Subst& subs)
	{
	    // std::cerr << "Decompile head, signature: " << fmap.get_data(s.first) << "/" << s.second << "\n";
		Functor h(s.first);
		if (s.second == 0) {
			return h;
		}

		// Make head
		std::vector<Functor> args(s.second,Functor());
		bool finished = false;

		for ( ; !finished; ++i) {
		    // std::cerr << "decompile head, instruction: " << *i << "\n";
			switch (i->opcode) {
			case instruction::GET_CON: args[i->a2] = Functor(i->a1); break;
			case instruction::GET_INT: args[i->a2] = Functor::make_int(i->i1); break;
			case instruction::GET_FLT: args[i->a2] = Functor::make_double(i->d1); break;
			case instruction::GET_VAR:
				args[i->a2] = Functor(Functor::unique_index());
				x2v.insert(std::make_pair(i->a1,args[i->a2].id())); 
				break;
			case instruction::GET_VARY:
				args[i->a2] = Functor(Functor::unique_index());
				y2v.insert(std::make_pair(i->a1,args[i->a2].id()));
				break;
			case instruction::GET_VAL:
				{
					auto at = x2v.find(i->a1);
					if (at == x2v.end()) {
						// Peephole optimization has been used
						args[i->a1] = args[i->a2] = Functor(Functor::unique_index());
						x2v.insert(std::make_pair(i->a1,args[i->a1].id()));
					} else {
						args[i->a2] = Functor(at->second);
					}
				}
				break;
			case instruction::GET_VALY:
				args[i->a2] = Functor(y2v.find(i->a1)->second);
				break;
			case instruction::GET_STR:
			case instruction::GET_LIS:
				{
					int reg;
					sign_t ss;
					if (i->opcode == instruction::GET_STR) {
						ss = sign_t(i->a1,i->a2);
						reg = i->a3;
					} else {
						ss = sign_t(pair_id,2);
						reg = i->a1;
					}
					std::vector<Functor> tmp(ss.second,Functor());
				    // std::cerr << "decompile GET_STR, " << fmap.get_data(i->a1) << "/" << i->a2 << "\n";
					//std::cerr << "tmp[0].get_int(): " << tmp[0].arity() << "\n";
					//auto ii = i;
					++i;
					for (int n = 0; n < ss.second; ++n,++i) {
					    // std::cerr << "decompile head, nested instruction: " << *i << "\n";
						switch (i->opcode) {
						case instruction::UNI_CON: tmp[n] = Functor(i->a1); break;
						case instruction::UNI_INT: tmp[n] = Functor::make_int(i->i1); break;
						case instruction::UNI_FLT: tmp[n] = Functor::make_double(i->d1); break;
						case instruction::UNI_VAR:
							tmp[n] = Functor(Functor::unique_index());
							x2v.insert(std::make_pair(i->a1,tmp[n].id()));
							break;
						case instruction::UNI_VARY:
							tmp[n] = Functor(Functor::unique_index());
							y2v.insert(std::make_pair(i->a1,tmp[n].id()));
							break;
						case instruction::UNI_VAL:
						case instruction::UNI_LOCAL_VAL:
							tmp[n] = x2v.find(i->a1)->second;
							break;
						case instruction::UNI_VALY:
						case instruction::UNI_LOCAL_VALY:
							tmp[n] = Functor(y2v.find(i->a1)->second);
							break;
						case instruction::UNI_VOID:
							tmp[n] = Functor(Functor::unique_index());
							break;
						default:
							std::cerr << "Error, instruction: " << *i << "\n";
							assert(false);
							exit(1);
						}
					}
					// Build
					Functor f(ss.first);
					for (auto&& a : tmp) f.steal(new Functor(std::move(a)));
					if (reg < s.second) {
						// argument
						args[reg] = std::move(f);
					} else {
						// Store substitution
						auto var = x2v.find(reg)->second;
					    // std::cerr << "Storing subs: x->v is " << reg << " -> " << var << " -> " << f << "\n";
						assert(!subs.get(var));
						subs.steal(var, new Functor(std::move(f)));
					}
					--i; // cancel effect of next ++i
				}
				break;
			case instruction::NOOP:
			case instruction::TRY_ME_ELSE:
			case instruction::RETRY_ME_ELSE:
			case instruction::TRUST_ME:
			case instruction::GET_LVL:
			case instruction::ALLOC:
				// Ignore
				break;
			default:
				--i; // do not advance i
				finished = true;
			}
		}

		for (auto&& a : args) {
			if (a.is_nil()) {
				h.steal(new Functor(Functor::unique_index()));
			} else {
				h.steal(new Functor(std::move(a)));
			}
		}

		subs.expand();
		h.subst(subs);
		return h;
	}

	// Make term from instructions
	Functor decompile(const sign_t& s, const instruction* i)
	{
		std::map<int,id_type> x2v,y2v;
		//std::cerr << "Decompile head...\n";
		Subst subs;
		Functor head = decompile_head(s,i,x2v,y2v,subs);
		std::vector<Functor> blv;
	    // std::cerr << "Decompile, head: " << head << "\n";
		if (std::prev(i)->opcode != instruction::PROCEED) {
			for (int k = 1; ; ++k) {
				Functor bl = decompile_bl(i,x2v,y2v,(k == 1 ? &head : nullptr),subs);
				if (bl.is_nil()) break;
				blv.push_back( std::move(bl) );
				const instruction& pi = *std::prev(i);
				if (pi.opcode == instruction::EXEC || pi.opcode == instruction::DYN_EXEC || pi.opcode == instruction::PROCEED) {
					break;
				}
			}
		}

		for (auto&& p : blv) head.body_push_back(new Functor(std::move(p)));
		subs.expand();
		head.subst(subs);
		return head;
	}

	// Make term from instructions (no head)
	Functor decompile(const instruction* i)
	{
		std::map<int,id_type> x2v,y2v;
		std::vector<Functor> blv;
		Subst subs;
	    // std::cerr << "Decompile only body...\n";

		for (;;) {
			Functor bl = decompile_bl(i,x2v,y2v,nullptr,subs);
		    // std::cerr << "  bl: " << bl << "\n";
			if (bl.is_nil()) break;
			blv.push_back( std::move(bl) );
			const instruction& pi = *std::prev(i);
			if (pi.opcode == instruction::EXEC || pi.opcode == instruction::DYN_EXEC || pi.opcode == instruction::PROCEED) {
				break;
			}
		}

		Functor ans(new Functor(if_id));
		for (auto&& p : blv) {
			ans.body_push_back(new Functor(std::move(p)));
		}
		subs.expand();
		ans.subst(subs);
	    // std::cerr << "  decompile only body done: " << ans << "\n";
		return ans;
	}


	bool Program::prompt(const qconstraints& qc, answer_type& answers, int& solc)
	{
		//std::cerr << "Prompt, interactive: " << interactive << "\n";
		//std::cerr << "Prompt, var_map: " << var_map.size() << "\n";

		// Note: If no variables then definitely success.
		// If variables, then success if not cyclic

		// Build persistent substitutions
		Subst sub;
		std::map<const lp::term*,id_type> vmap;
		bool is_cyclic = false; // assume non-cyclic solution
		for (auto i = var_map.begin(); i != var_map.end(); ++i) {
			assert(!sub.get(i->first));
			try {
				sub.steal(i->first, term2functor(i->second,vmap));
			} catch (cyclic_term) {
				DEBUG_WARNING(std::cerr << "Warning: skipping cyclic solution " << Functor::get_data(i->first) << " = " << *i->second << "\n");
				is_cyclic = true;
				break;
			}
			//sub.insert(subst::value_type(i->first, std::unique_ptr<Functor>(lp::copy(t,venum))));
		}

		// Increase solution counter
		if (!is_cyclic) ++solc;

		// Print if interactive
		if (qc.interactive) {
			if (is_cyclic) {
				std::cout << "Cyclic.\n";
			} else if (sub.is_empty()) {
				std::cout << "Yes.\n";
			} else {
				// Strip off all anonymous bindings
				Subst scopy = sub;
				scopy.expand();
				for (auto i = scopy.begin(); i != scopy.end(); ) {
					const Functor* bound_to = i->second.first;
					if (bound_to->is_variable()) {
						const data& dat = Functor::get_data(bound_to->id());
						if (!dat.is_atom() || dat.get_atom()[0] == '_') {
							i = scopy.erase(i);
						} else ++i;
					} else ++i;
				}
				if (scopy.is_empty()) {
					std::cout << "Yes.\n";
				} else {
					std::cout << "Found solution:\n";
					for (const auto& p : scopy) {
						std::cout << lp::fmap.get_data(p.first) << " = " << *p.second.first << "\n";
					}
				}
			}
		}

		// Store persistent substitutions
		if (!is_cyclic) {
			answers.push_back(std::move(sub));
		}

		if (int(answers.size()) >= qc.recall) {
			if (qc.interactive) {
				DEBUG_TRACE(std::cerr << "reached recall limit " << qc.recall << "\n");
			}
			return false; // don't continue
		}

		assert(env_back >= stack_bottom);
		const bool is_backtrackable = (env_back != stack_bottom); 

		if (qc.interactive && is_backtrackable) {
			std::cout << "more? ";
			std::string s;
			for (;;) {
				std::cin.sync();
				std::getline(std::cin,s);
				if (s == ".") {
					return false;
				} else if (s == ";") {
					backtrack();
					return true;
				} else {
					std::cout << "incorrect input, use ; or .\n";
				}
			}
		} else {
			// Non-interactive: backtrack since recall hasn't been reached
			backtrack();
			return true;
		}
	}

	// Initialize builtins
	builtin_lookup_type define_builtins()
	{
		// Define builtins
		using namespace lp;
		builtin_lookup_type b;

		// Logical
		//b[make_pair("or",2)] = &Program::builtin_or;
		b[make_pair(true_id,0)] = instruction::NOOP;
		b[make_pair(fail_id,0)] = instruction::FAIL;
		b[make_pair(false_id,0)] = instruction::FAIL;
		// Functor
		b[make_pair(var_id,1)] = instruction::VAR;
		b[make_pair(nonvar_id,1)] = instruction::NONVAR;
		b[make_pair(constant_id,1)] = instruction::ATOMIC; //Progol synonym for atomic/1
		b[make_pair(atom_id,1)] = instruction::ATOM;
		b[make_pair(atomic_id,1)] = instruction::ATOMIC;
		b[make_pair(number_id,1)] = instruction::NUMBER;
		b[make_pair(integer_id,1)] = instruction::INTEGER;
		b[make_pair(int_id,1)] = instruction::INTEGER; // synonym for integer/1
		b[make_pair(float_id,1)] = instruction::FLOAT;
		b[make_pair(equal_id,2)] = instruction::UNIFY;
		b[make_pair(equal2_id,2)] = instruction::UNIFY; // '_='/2, used for variable splitting
		b[make_pair(unequal_id,2)] = instruction::NONUNIFIABLE;
		// Term ordering
		b[make_pair(syn_equal_id,2)] = instruction::EQUAL;
		b[make_pair(syn_unequal_id,2)] = instruction::UNEQUAL;
		b[make_pair(lessat_id,2)] = instruction::BEFORE;
		b[make_pair(lesseqat_id,2)] = instruction::BEFORE_EQ;
		b[make_pair(greaterat_id,2)] = instruction::AFTER;
		b[make_pair(greatereqat_id,2)] = instruction::AFTER_EQ;
		b[make_pair(compare_id,3)] = instruction::COMPARE;
		b[make_pair(isvariant_id,2)] = instruction::VARIANT;
		b[make_pair(isnotvariant_id,2)] = instruction::NOT_VARIANT;

		b[make_pair(univ_id,2)] = instruction::UNIV;
		b[make_pair(functor_id,3)] = instruction::FUNCTOR;
		b[make_pair(arg_id,3)] = instruction::ARG;
		b[make_pair(name_id,2)] = instruction::NAME;
		// Numeric
		b[make_pair(is_id,2)] = instruction::IS;
		b[make_pair(math_equal_id,2)] = instruction::NUMEQUAL;
		b[make_pair(math_unequal_id,2)] = instruction::NUMUNEQUAL;
		b[make_pair(lt_id,2)] = instruction::NUMLESS;
		b[make_pair(le_id,2)] = instruction::NUMLESSEQ;
		b[make_pair(gt_id,2)] = instruction::NUMGREATER;
		b[make_pair(ge_id,2)] = instruction::NUMGREATEREQ;
		//// Procedural
		//b[make_pair(sort_id,2)] = instruction::SORT;
		//// Side effects
		b[make_pair(get_id,2)] = instruction::GET_PARAM;
		b[make_pair(set_id,2)] = instruction::SET_PARAM;
		b[make_pair(halt_id,0)] = instruction::HALT;
		b[make_pair(op_id,3)] = instruction::OP;
		b[make_pair(listing_id,0)] = instruction::LISTING;
		b[make_pair(listing_wam_id,0)] = instruction::LISTING_WAM;
		b[make_pair(listing_id,1)] = instruction::LISTING1;
		b[make_pair(listing_wam_id,1)] = instruction::LISTING_WAM1;
		//b[make_pair(display_id,1)] = instruction::DISPLAY;
		b[make_pair(write_id,1)] = instruction::WRITE;
		b[make_pair(nl_id,0)] = instruction::NL;
		b[make_pair(consult_id,1)] = instruction::CONSULT;
		b[make_pair(pair_id,2)] = instruction::CONSULT_LIST; // loading files with [file1,file2]
		b[make_pair(assert_id,1)] = instruction::ASSERT;
		b[make_pair(assertz_id,1)] = instruction::ASSERT;
		b[make_pair(asserta_id,1)] = instruction::ASSERTA;
		b[make_pair(nb_linkarg_id,3)] = instruction::NB_LINKARG;
		b[make_pair(duplicate_term_id,2)] = instruction::DUPE_TERM;
		//b[make_pair(retract_id,1)] = instruction::RETRACT;
		b[make_pair(retractall_id,1)] = instruction::RETRACTALL;
		b[make_pair(modeh_id,2)] = instruction::MODEH;
		b[make_pair(modeb_id,2)] = instruction::MODEB;
		b[make_pair(modeb_id,3)] = instruction::MODEB3; // extended mode declaration with max_occurs/functional
		b[make_pair(implies_id,2)] = instruction::IMPLIES;
		b[make_pair(prevents_id,2)] = instruction::PREVENTS;
		b[make_pair(prevent_id,1)] = instruction::PREVENT;
		b[make_pair(determination_id,2)] = instruction::DETERMINATION;
		b[make_pair(include_predicate_id,2)] = instruction::DETERMINATION;
		b[make_pair(exclude_predicate_id,2)] = instruction::EXCLUDE_PRED;
		//// (builtin instead) b[make_pair(findall_id,3)] = &Program::builtin_findall;

		//// ILP
		//// b[make_pair(compress_id,0)] = &Program::builtin_compress;
		//b[make_pair(bottom_id,2)] = instruction::BOTTOM;
		b[make_pair(generalize_id,0)] = instruction::GENERALIZE0; // use dpll
		b[make_pair(generalize_id,1)] = instruction::GENERALIZE1; // select method
		b[make_pair(generalise_id,0)] = instruction::GENERALIZE0; // brittish spelling
		b[make_pair(generalise_id,1)] = instruction::GENERALIZE1; // brittish spelling
		//// Stochastic
		//// b[make_pair(sample_id,2)] = &Program::builtin_sample; TODO: must use Modeh?
		//b[make_pair(label_id,1)] = instruction::LABEL;

		//cerr << "BUILTIN done\n";
		return b;
	};
	// Set static builtin table
	builtin_lookup_type builtin = lp::define_builtins();

	//void wam_print(std::ostream& os)
	//{
	//	int line = 0;
	//	os << "HS:\n";
	//	for (int k = 0; k < 100 && lp::heap[k].get_type(); ++k, ++line) {
	//		os << "  " << line << ": " << lp::heap[k] << "\n";
	//	}
	//	line = 1; // variables start at index 1
	//	os << "Var Regs:\n";
	//	for (int k = 1; k < 10; ++k, ++line) {
	//		os << "  " << line << ": " << *var_regs[ k] << "\n";
	//	}
	//}


	//========================= Wam Helper Functions ===========================//

	void bind(term* x, term* y)
	{
		assert( x->is_variable() );
		assert( !y->is_variable() || (is_heap(x) && is_heap(y) && x < y) 
			|| (is_stack(x) && is_stack(y) && x < y) || (is_stack(x) && is_heap(y)) );
		//std::cerr << "Binding: "; x->print(cerr); cerr << " => "; y->print(cerr); cerr << "\n";
		//std::cerr << "inside bind(), checking for self cycles...\n";
		//assert( !is_cyclic(y) );
		//std::cerr << "inside bind(), checking for other cycles...\n";
		//assert( !is_cyclic(x,y) );
		//std::cerr << "inside bind(), NO cycles\n";
		x->set_ptr(y);
		update_trail(x);
	}

	void update_trail(term* p)
	{
		assert( !p->is_function() );
		bool do_trail = false;
		if (lp::is_heap(p)) {
			// p before heap[heap_back] but points to after => reference must be undone
			if (p < &heap[heap_back]) do_trail = true;
		} else {
			assert( lp::is_stack(p) );
			if (p < &lp::stack[env_back]) do_trail = true;
		}
		if (do_trail) {
			trail[tindex] = p;
			//std::cerr << "update_trail: " << *trail[tindex] << "\n";
			assert( !trail[tindex]->is_function() );
			++tindex;
		}
	}

	void unwind_trail(int x, int y)
	{
		assert(x <= y && x <= tindex && y <= tindex);
		//std::cerr << "unwind_trail: " << x << " -> " << y << "\n";
		for ( ; x < y; ++x) {
			// note: trail can be VAR, CON, INT, or FLT
			assert( trail[x]->is_variable() );
			//std::cerr << "  unwinding " << trail[x] << "\n";
			trail[x]->set_ptr( trail[x] );
			//*trail[x] = term::make_variable( trail[x]);
		}
	}

	void tidy_trail()
	{
	    //std::cerr << "Tidy trail, env_back: " << env_back << "\n";
	    //std::cerr << "  stack[" << env_back + lp::stack[env_back].arity() + 3 << "]\n";
		int i = lp::stack[env_back + lp::stack[env_back].arity() + 3].get_fun();
	    //std::cerr << "  i: " << i << ", tindex: " << tindex << "\n";
		while (i < tindex) {
			bool tidy = false;
			assert(trail[i] != nullptr);
			if (lp::is_heap(trail[i])) {
				if (trail[i] < &lp::heap[heap_back]) tidy = true;
			} else {
				assert( lp::is_stack(trail[i]) || !(cerr << "trail: " << *trail[i] << "\n") );
				if (trail[i] < &lp::stack[env_back]) tidy = true; 
			}
			if (tidy) {
				trail[i] = trail[tindex - 1];
				--tindex;
			} else {
				++i;
			}
		}
	}

	bool unify(term* x, term* y)
	{
		//std::set<const term*> fun1,fun2; // to detect cyclic terms
		if (is_cyclic(x)) {
			//std::cerr << "unify(x,y): x is cyclic\n";
			return false;
		} else if (is_cyclic(y)) {
			//std::cerr << "unify(x,y): y is cyclic\n";
			return false;
		}

		std::stack<term*> pdl;
		//std::cerr << "unification: "; x->print(cerr); cerr << " <-> "; y->print(cerr); std::cerr << "\n";
		//cerr << " (" << x << " <-> " << y << ")\n";
		pdl.push(y);
		pdl.push(x);
		// TODO: make max_unification adjustable parameter
		const int MAX_UNIFICATIONS = std::numeric_limits<int>::max(); //1000000; 
		int i = 0;
		for ( ; !pdl.empty() && i < MAX_UNIFICATIONS; ++i) {
			term* d1 = deref(pdl.top());
			pdl.pop();
			term* d2 = deref(pdl.top());
			pdl.pop();
			//std::cerr << "  dereferenced: "; d1->print(cerr); cerr << " <-> "; d2->print(cerr); cerr << "\n";
			if (d1 != d2) {
				if (d1->is_variable()) {
					bool d1_to_d2 = true;
					if (d2->is_variable()) {
						if (is_heap(d1)) {
							if (is_stack(d2) || d2 < d1) d1_to_d2 = false;
						} else {
							assert(is_stack(d1));
							if (is_stack(d2) && d2 < d1) d1_to_d2 = false;
						}
					}
					if (d1_to_d2) {
						// d1 -> d2
						//if (is_cyclic(d1,d2)) return false;
						bind(d1,d2);
					} else {
						// d2 -> d1
						//if (is_cyclic(d2,d1)) return false;
						bind(d2,d1);
					}
				} else {
					switch (d2->get_type()) {
					case term::VAR: case term::EVAR:
						assert( !d1->is_variable() );
						// d2 -> d1
						//if (is_cyclic(d2,d1)) return false;
						bind(d2,d1);
						break;
					case term::CON:
						if (!d1->is_constant() || d1->arity() != d2->arity()) return false;
						break;
					case term::INT:
						if (!d1->is_int() || d1->get_int() != d2->get_int()) return false;
						break;
					case term::FLT:
						if (!d1->is_double() || d1->get_double() != d2->get_double()) return false;
						break;
					case term::LIS:
						if (!d1->is_list()) return false;
						pdl.push(d2 + 2);
						pdl.push(d1 + 2);
						pdl.push(d2 + 1);
						pdl.push(d1 + 1);
						break;
					default:
						assert(d2->is_function() && !d2->is_list());
						if (!d1->is_function() || d1->is_list()) return false;
						if (d1->get_type() != d2->get_type() || d1->arity() != d2->arity()) {
							return false;
						}
						// Push all arguments if functors
						for (int k = d1->arity(); k > 0; --k) {
							//std::cerr << "pushing back functor arg: "; (d2+k)->print(cerr); cerr << "\n";
							pdl.push(d2 + k);
							//std::cerr << "pushing back functor arg: "; (d1+k)->print(cerr); cerr << "\n";
							pdl.push(d1 + k);
						}
						break;
						//std::cerr << "Error: unrecognized type\n";
						//assert(false);
						//exit(1);
					}
				}
			}
		}

		if (i >= MAX_UNIFICATIONS) {
			DEBUG_WARNING(std::cerr << "Warning: max unification " << MAX_UNIFICATIONS << " reached (probably cyclic term)\n");
			return false;
		}
		return true;
	}

	bool unify_external(term* x, term* y, std::map<term*,term*>& binds)
	{
		std::stack<term*> pdl;
		auto follow = [&](term* t) -> term* {
			std::map<term*,term*>::iterator at;
			for (;;) {
				if (t->is_variable()) {
					if (t != t->get_ptr()) t = t->get_ptr();
					else if ((at = binds.find(t)) != binds.end()) t = at->second;
					else break;
				} else break;
			}
			return t;
		};

		pdl.push(y);
		pdl.push(x);
		while (!pdl.empty()) {
			term* d1 = follow(pdl.top());
			pdl.pop();
			term* d2 = follow(pdl.top());
			pdl.pop();
			//std::cerr << "  dereferenced: "; d1->print(cerr); cerr << " <-> "; d2->print(cerr); cerr << "\n";
			if (d1 != d2) {
				if (d1->is_variable()) {
					// d1 -> d2
					//if (is_cyclic(d1,d2,binds)) return false;
					binds.insert(std::make_pair(d1,d2));
				} else {
					switch (d2->get_type()) {
					case term::VAR: case term::EVAR:
						assert( !d1->is_variable() );
						// d2 -> d1
						//if (is_cyclic(d2,d1,binds)) return false;
						binds.insert(std::make_pair(d2,d1));
						break;
					case term::CON:
						if (!d1->is_constant() || d1->arity() != d2->arity()) return false;
						break;
					case term::INT:
						if (!d1->is_int() || d1->get_int() != d2->get_int()) return false;
						break;
					case term::FLT:
						if (!d1->is_double() || d1->get_double() != d2->get_double()) return false;
						break;
					case term::LIS:
						if (!d1->is_list()) return false;
						pdl.push(d2 + 1);
						pdl.push(d1 + 1);
						pdl.push(d2 + 2);
						pdl.push(d1 + 2);
						break;
					default:
						assert(d2->is_function());
						if (!d1->is_function()) return false;
						if (d1->get_type() != d2->get_type() || d1->arity() != d2->arity()) {
							return false;
						}
						// Push all arguments if functors
						for (int k = d1->arity(); k > 0; --k) {
							//std::cerr << "pushing back functor arg: "; (d2+k)->print(cerr); cerr << "\n";
							pdl.push(d2 + k);
							//std::cerr << "pushing back functor arg: "; (d1+k)->print(cerr); cerr << "\n";
							pdl.push(d1 + k);
						}
						break;
						//std::cerr << "Error: unrecognized type\n";
						//assert(false);
						//exit(1);
					}
				}
			}
		}
		return true;
	}


	void Program::put_unsafe_value(int yi, int a)
	{
		term* addr = deref( &lp::stack[env_i + yi + 3] );
		//std::cerr << "put_unsafe_value: Y" << yi << " => " << *addr << "\n";
		if (is_stack(addr)) {
			// Globalize binding
			assert(addr->is_variable());
			lp::heap[heap_i] = term::make_variable(&lp::heap[heap_i]);
			bind(addr,&lp::heap[heap_i]);
		    //std::cerr << "  put_unsafe, addr now points to: " << *deref(addr) << "\n";
			var_regs[a] = &lp::heap[heap_i];
			++heap_i;
		} else {
			return put_stack_value(yi,a);
		}
		++cap;
	}


	void Program::set_local_value(int v)
	{
		// Check if v points to something on the stack, in which case we need to globalize
		term* addr = deref(var_regs[v]);
		if (lp::is_stack(addr)) {
			// Globalize
			lp::heap[heap_i] = term::make_variable(&lp::heap[heap_i]);
			bind(addr,&lp::heap[heap_i]);
			++heap_i;
		} else {
			// Already globalized, just reference
			lp::heap[heap_i] = term::make_variable(addr); // REF?
			++heap_i;
		}
		++cap;
	}

	void Program::set_local_stack_value(int yi)
	{
		// No stack-to-stack ref - stack variable should point to heap variable
		term* addr = deref(&lp::stack[env_i + yi + 3]);
		if (lp::is_stack(addr)) {
		    //std::cerr << "set_local_stack_value: " << *addr << "\n";
			// Globalize
			if (addr->is_variable()) {
				lp::heap[heap_i] = term::make_variable(&lp::heap[heap_i]);
			} else {
				assert( !addr->is_function() );
				lp::heap[heap_i] = *addr;
				*addr = term::make_variable(addr);
			}
			bind(addr,&lp::heap[heap_i]);
			++heap_i;
		} else {
			// Already globalized, just reference
			lp::heap[heap_i] = term::make_variable(addr); // REF?
			++heap_i;
		}
		++cap;
	}


	void Program::set_void(int n)
	{
		for (int k = heap_i; k <= heap_i+n-1; ++k) {
			lp::heap[k] = term::make_variable(&lp::heap[k]);
		}
		heap_i += n;
		++cap;
	}


	void Program::get_structure(int f, int n, int t)
	{
		bool fail = false;
		term* addr = deref(var_regs[t]);
		if (addr->is_variable()) {
			lp::heap[heap_i] = term(f,n);
		    //std::cerr << "get_str, binding " << *addr << " to "; lp::heap[heap_i].print(cerr); cerr << "\n";
			bind(addr,&lp::heap[heap_i]);
			++heap_i;
			wam_mode = WRITE;
		} else if (addr->is_function(f,n)) {
			s_ptr = addr + 1;
			wam_mode = READ;
		} else {
			fail = true;
		}

		if (fail) {
			backtrack();
		} else {
			++cap;
		}
	}

	void Program::get_list(int t)
	{
		bool fail = false;
		term* addr = deref(var_regs[t]);
		if (addr->is_variable()) {
			lp::heap[heap_i] = term::make_list();
		    //std::cerr << "get_str, binding " << *addr << " to "; lp::heap[heap_i].print(cerr); cerr << "\n";
			bind(addr,&lp::heap[heap_i]);
			++heap_i;
			wam_mode = WRITE;
		} else if (addr->is_list()) {
			//std::cerr << "  get_list: " << *addr << "\n1:  " << *(addr+1) << "\n2:  " << *(addr+2) << "\n";
			s_ptr = addr + 1;
			wam_mode = READ;
		} else {
			fail = true;
		}

		if (fail) {
			backtrack();
		} else {
			++cap;
		}
	}

	void Program::get_constant(int c, int t)
	{
		bool fail = false;
		//std::cerr << "get_constant, X[" << t << "] = " << ", c = " << fmap.get_data(c) << "\n";
		assert( var_regs[t] != nullptr );
		//std::cerr << "calling deref:\n"; /*print_all(cerr,var_regs[t]);*/
		term* addr = deref(var_regs[t]);
		//cerr << "deref done\n";
		//cerr << "  c==X[" << t << "]: " << fmap.get_data(c) << " == "; cerr << *addr << "\n";
		switch (addr->get_type()) {
		case term::VAR: case term::EVAR:
			assert( addr->get_ptr() == addr );
			//std::cerr << "  get_constant, VAR\n";
			//bind(addr,c); // bind to constant
			lp::heap[heap_i] = term::make_constant(c);
			bind(addr,&lp::heap[heap_i]);
			++heap_i;
			//std::cerr << "get_constant: " << *addr << " => " << *deref(addr) << "\n";
			break;
		case term::CON:
			//std::cerr << "  get_constant, CON: " << fmap.get_data(addr->arity()) << " == " << fmap.get_data(c) << "\n";
			fail = (c != addr->arity());
			break;
		default:
			//std::cerr << "  get_constant, fail\n";
			fail = true;
		}

		if (fail) {
			backtrack();
		} else {
			++cap;
		}
	}

	void Program::get_int(long long v, int t)
	{
		bool fail = false;
		term* addr = deref(var_regs[t]);
		//std::cerr << "get_int, addr type: " << addr->get_type() << " (" << *addr << ")\n";
		switch (addr->get_type()) {
		case term::VAR: case term::EVAR:
			//std::cerr << "get_int, VAR: " << v << " == " << *addr << "\n";
			lp::heap[heap_i] = term::make_int(v);
			bind(addr,&lp::heap[heap_i]);
			++heap_i;
			//std::cerr << "get_constant: " << *addr << " => " << *deref(addr) << "\n";
			break;
		case term::INT:
			//std::cerr << "get_int, INT: " << v << " == " << addr->get_int() << "\n";
			fail = (v != addr->get_int());
			break;
		default: 
			// std::cerr << "get_constant, type: " << addr->get_type() << "\n";
			fail = true;
		}

		if (fail) {
			backtrack();
		} else {
			++cap;
		}
	}

	void Program::get_float(double v, int t)
	{
		bool fail = false;
		term* addr = deref(var_regs[t]);
		switch (addr->get_type()) {
		case term::VAR: case term::EVAR:
			lp::heap[heap_i] = term::make_double(v);
			bind(addr,&lp::heap[heap_i]);
			++heap_i;
			//std::cerr << "get_constant: " << *addr << " => " << *deref(addr) << "\n";
			break;
		case term::FLT:
			fail = (v != addr->get_double());
			break;
		default: 
			// std::cerr << "get_constant, type: " << addr->get_type() << "\n";
			fail = true;
		}

		if (fail) {
			backtrack();
		} else {
			++cap;
		}
	}

	void Program::unify_variable(int v)
	{
		switch (wam_mode) {
		case READ:
			var_regs[v] = s_ptr;
			++s_ptr;
			break;
		case WRITE:
			return set_variable(v);
		}
		++cap;
	}

	void Program::unify_stack_variable(int yi)
	{
		switch (wam_mode) {
		case READ:
			// std::cerr << "unify stack variable address: " << &lp::stack[env_i + yi + 3] << "\n";
			lp::stack[env_i + yi + 3] = term::make_variable(s_ptr);
			++s_ptr;
			break;
		case WRITE:
			return set_stack_variable(yi);
		}
		++cap;
	}

	void Program::unify_value(int v)
	{
		switch (wam_mode) {
		case READ:
			if (!unify(var_regs[v],s_ptr)) {
				backtrack();
				return;
			}
			++s_ptr;
			break;
		case WRITE:
			return set_value(v);
		}
		++cap;
	}

	void Program::unify_stack_value(int yi)
	{
		switch (wam_mode) {
		case READ:
			//assert( lp::stack[eindex + yi + 1].is_ref() );
			if (!unify(lp::stack[env_i + yi + 3].get_ptr(),s_ptr)) {
				backtrack();
				return;
			}
			++s_ptr;
			break;
		case WRITE:
			return set_stack_value(yi);
		}
		++cap;
	}

	void Program::unify_local_value(int v)
	{
		switch (wam_mode) {
		case READ:
			if (!unify(var_regs[v],s_ptr)) {
				backtrack();
				return;
			}
			++s_ptr;
			break;
		case WRITE:
			return set_local_value(v);
		}
		++cap;
	}

	void Program::unify_local_stack_value(int yi)
	{
		switch (wam_mode) {
		case READ:
			if (!unify(lp::stack[env_i + yi + 3].get_ptr(),s_ptr)) {
				backtrack();
				return;
			}
			++s_ptr;
			break;
		case WRITE:
			return set_local_stack_value(yi);
		}
		++cap;
	}

	void Program::unify_constant(int c)
	{
		switch (wam_mode) {
		case READ:
			{
				term* addr = deref(s_ptr);
				//std::cerr << "  unify_constant: sptr: " << *s_ptr << " => " << *addr << "\n"; 
				switch (addr->get_type()) {
				case term::VAR: case term::EVAR:
					lp::heap[heap_i] = term::make_constant(c);
					bind(addr,&lp::heap[heap_i]);
					++heap_i;
					break;
				case term::CON:
					//std::cerr << "  as ID: " << fmap.get_data(addr->get_const()) << " == " << fmap.get_data(c) << "\n"; 
					if (addr->get_const() == c) {
						break;
					}
					// else: fall through
				default:
					backtrack();
					return;
				}
				++s_ptr;
			}
		case WRITE:
			return set_constant(c);
		}
		++cap;
	}

	void Program::unify_int(long long v)
	{
		switch (wam_mode) {
		case READ:
			{
				term* addr = deref(s_ptr);
				switch (addr->get_type()) {
				case term::VAR: case term::EVAR:
					lp::heap[heap_i] = term(v);
					bind(addr,&lp::heap[heap_i]);
					++heap_i;
					break;
				case term::INT:
					if (addr->get_int() == v) {
						break;
					}
					// else: fall through
				default:
					backtrack();
					return;
				}
				++s_ptr;
			}
		case WRITE:
			return set_int(v);
		}
		++cap;
	}

	void Program::unify_float(double v)
	{
		switch (wam_mode) {
		case READ:
			{
				term* addr = deref(s_ptr);
				switch (addr->get_type()) {
				case term::VAR: case term::EVAR:
					lp::heap[heap_i] = term(v);
					bind(addr,&lp::heap[heap_i]);
					++heap_i;
					break;
				case term::FLT:
					if (addr->get_double() == v) {
						break;
					}
					// else: fall through
				default:
					backtrack();
					return;
				}
				++s_ptr;
			}
		case WRITE:
			return set_float(v);
		}
		++cap;
	}

	void Program::unify_void(int n)
	{
		switch (wam_mode) {
		case READ:
			//std::cerr << "unify_void, n=" << n << "\n  sptr: " << *s_ptr << "\n  sptr+1: " << *(s_ptr+1) << "\n";
			s_ptr += n;
			break;
		case WRITE:
			return set_void(n);
		}
		++cap;
	}

	void Program::reg_hvar(int id)
	{
		const auto s = var_map.insert(std::make_pair(id,&lp::heap[heap_i-1]));
		assert( s.second );
		//std::cerr << "reg_hvar: " << Functor::get_data(id) << " => " << &lp::heap[heap_i-1] << " => " << lp::heap[heap_i-1] << "\n";
		++cap;
	}

	void Program::reg_svar(int id, int yi)
	{
		const auto s = var_map.insert(std::make_pair(id,&lp::stack[env_i + yi + 3]));
		assert( s.second );
		++cap;
	}

	void Program::globalize_svars()
	{
		// TODO: more efficient implementation?
		for (auto& p : var_map) {
			if (!lp::is_stack(p.second)) continue;
			term* t = deref(p.second);
			if (lp::is_stack(t)) {
				// VAR/CON/INT/FLT still on stack
				if (t->is_variable()) {
					assert(t->get_ptr() == t);
					lp::heap[heap_i] = term::make_variable(&lp::heap[heap_i]);
				} else {
					assert(!t->is_function());
					lp::heap[heap_i] = *t;
				}
			} else {
				lp::heap[heap_i] = term::make_variable(t);
			}
			t = &lp::heap[heap_i];
			//std::cerr << "Globalize, t: " << *t << " [" << lp::is_stack(t) << "]\n";
			++heap_i;
		}
		++cap;
	}

	void Program::reg_arg(int k)
	{
		assert(k >= 0);
		if (int(arg_map.size()) <= k) arg_map.resize(k+1);
		arg_map[k] = var_regs[k];
		++cap;
	}


	void Program::allocate()
	{
		// std::cerr << "allocate, env_i: " << env_i << ", env_back: " << env_back << "\n";
		int i;
		if (env_i >= env_back) {
			// Environment Frame
			if (cap2.line == 0) {
				i = env_i + 3; // place i at beginning of our frame
			} else {
				const instruction& call_instr = *std::prev(cap2);
				switch (call_instr.opcode) {
				case instruction::CALL:
					i = env_i + call_instr.a3 + 3; // place i at beginning of our frame
					break;
				case instruction::DYN_CALL:
				case instruction::DYN_STACK_CALL:
					i = env_i + call_instr.a2 + 3; // place i at beginning of our frame
					break;
				default: 
				    std::cerr << "ERROR: expected CALL or DYN_CALL\n";
					assert(false);
					exit(1);
				}
			}
		} else {
			// Choice Point Frame
			i = env_back + lp::stack[env_back].arity() + 7; // place i at beginning of our frame
		}
	    // std::cerr << "Allocate, starts at " << &lp::stack[i] << "\n";
		lp::stack[i] = term(env_i,cap2.line);
	    // std::cerr << "  allocate, storing cap2.con_ptr at " << &lp::stack[i+1] << "\n";
		if (cap2.con_ptr) {
			lp::stack[i+1] = term(cap2.cl_i, reinterpret_cast<long long>(cap2.con_ptr));
		} else {
		    // std::cerr << "  allocate, cap2.con_ptr is nullptr\n";
			assert(reinterpret_cast<long long>(cap2.con_ptr) == 0);
			lp::stack[i+1] = term(reinterpret_cast<int>(cap2.cl_ptr), reinterpret_cast<long long>(cap2.con_ptr));
		}
		lp::stack[i+2] = term(qlevel,0);
	    // std::cerr << "  Allocate frame end at " << &lp::stack[i+3] << " + number of Y-variables\n";
		env_i = i;
		++cap;
	}

	void Program::deallocate()
	{
		qlevel = lp::stack[env_i+2].get_fun();
		//assert(qlevel == lp::stack[env_i+2].get_fun() 
		//	|| !(std::cerr << "qlevel: " << qlevel << ", stored: " << lp::stack[env_i+2].get_fun() << "\n"));
	    // std::cerr << "Deallocate, retrieving cap2.con_ptr from " << &lp::stack[env_i+1] << "\n";
		cap2.con_ptr = reinterpret_cast<concept*>(lp::stack[env_i+1].get_int());
		if (cap2.con_ptr) {
		    // std::cerr << "  deallocate, cap2.con_ptr is NOT nullptr\n";
			cap2.cl_i = lp::stack[env_i+1].get_fun();
		} else {
		    // std::cerr << "  deallocate, cap2.con_ptr is nullptr\n";
			cap2.cl_ptr = reinterpret_cast<decltype(cap2.cl_ptr)>(lp::stack[env_i+1].get_fun());
		}
		cap2.line = lp::stack[env_i].arity();
		env_i = lp::stack[env_i].get_fun();
		// std::cerr << "deallocate, restoring env_i: " << env_i << ", env_back: " << env_back << "\n";
		//std::cerr << "  Deallocate, X1: " << *var_regs[1] << " => " << *deref(var_regs[1]) << "\n";
		++cap;
	}

	void Program::call(int f, int n, int vars)
	{
		//std::cerr << "  In call " << vars << "\n" << "  " << "X1: " << *deref(var_regs[0]) << "\n"; 
		auto at = kb.find(std::make_pair(f,n));
		if (at == kb.end()) {
			const auto d = this->params.force_int(parameters::require_defs);
			if (d == 1) {
				DEBUG_WARNING(std::cerr << "Warning: Missing definition of " << fmap.get_data(f) << "/" << n << "\n");
				throw wam_fail();
			} else if (d > 1) {
				DEBUG_WARNING(std::cerr << "Error: Missing definition of " << fmap.get_data(f) << "/" << n << "\n");
				throw destruct_and_exit();
			} else {
				backtrack();
			}
		} else {
			cap2 = std::next(cap);
			qlevel2 = qlevel;
			num_of_args = n;
			env_back0 = env_back;
			cap.reset(&at->second);
			++qlevel;
			// std::cerr << "call, setting cap to: " << *cap << "\n";
		}
	}

	void Program::execute(lp::id_type f, int n)
	{
		//std::cerr << "In exec\n" << "  " << "X1: " << *deref(var_regs[1]) << "\n  X2: " << *deref(var_regs[2]) << "\n"; 
		auto at = kb.find(std::make_pair(f,n));
		if (at == kb.end()) {
			const auto d = this->params.force_int(parameters::require_defs);
			if (d == 1) {
				DEBUG_WARNING(std::cerr << "Warning: Missing definition of " << fmap.get_data(f) << "/" << n << "\n");
				throw wam_fail();
			} else if (d > 1) {
				DEBUG_WARNING(std::cerr << "Error: Missing definition of " << fmap.get_data(f) << "/" << n << "\n");
				throw destruct_and_exit();
			} else {
				backtrack();
			}
		} else {
			num_of_args = n;
			env_back0 = env_back;
			cap.reset(&at->second);
			++qlevel;
		}
	}

	void Program::dynamic_call(int v, int vars)
	{
		term* f = deref(var_regs[v]);
		// Load all arguments into registries
	    //std::cerr << "Dynamic call: " << *f << "\n";
		int k = 0;
		for (auto a = f->arg_begin(); a != f->arg_end(); ++a,++k) {
			assert(k <= v);
			var_regs[k] = &*a;
		    //std::cerr << "  dynamic call, arg " << k << ": " << *var_regs[k] << "\n";
		}
		const sign_t s = f->signature();
		return this->call(s.first,s.second,vars);
	}

	void Program::dynamic_stack_call(int yi, int vars)
	{
		term* f = deref(lp::stack[env_i + yi + 3].get_ptr());
		// Load all arguments into registries
	    //std::cerr << "Dynamic_stack_call: " << *f << " (" << f << ")\n";
		int k = 1;
		for (auto a = f->arg_begin(); a != f->arg_end(); ++a,++k) {
			var_regs[k] = &*a;
		    //std::cerr << "  dynamic_stack_call, arg " << k << ": " << *var_regs[k] << "\n";
		}
		const sign_t s = f->signature();
		return this->call(s.first,s.second,vars);
	}

	void Program::dynamic_execute(int v)
	{
		term* f = deref(var_regs[v]);
		// Load all arguments into registries
		int k = 0;
		//std::cerr << "Dynamic_execute X" << v << ": " << *f << "\n";
		for (auto a = f->arg_begin(); a != f->arg_end(); ++a,++k) {
			assert(k <= v);
			var_regs[k] = &*a;
		}
		const sign_t s = f->signature();
		return this->execute(s.first,s.second);
	}

	void Program::dynamic_stack_execute(int yi)
	{
		term* f = deref(lp::stack[env_i + yi + 3].get_ptr());
		// Load all arguments into registries
		int k = 1;
		for (auto a = f->arg_begin(); a != f->arg_end(); ++a,++k) {
			var_regs[k] = &*a;
		}
		const sign_t s = f->signature();
		return this->execute(s.first,s.second);
	}

	void Program::backtrack()
	{
		assert(env_back >= stack_bottom);
		if (env_back == stack_bottom) {
			// fail and exit
			throw wam_fail();
		} else {
			// std::cerr << "backtrack, env_back: " << env_back << "\n";
			//const int f = lp::stack[env_back].get_fun();
			const int n = lp::stack[env_back].arity();
			env_back0 = lp::stack[env_back + n + 2].arity();
			// std::cerr << "  setting cap.cl_i from " << cap.cl_i << " to ";
			cap.con_ptr = reinterpret_cast<concept*>(lp::stack[env_back + n + 5].get_int());
			if (cap.con_ptr) {
				cap.cl_i = lp::stack[env_back + n + 5].get_fun();
			} else {
				// TODO: all these cases can be removed since it is allowed to convert type using union
				cap.cl_ptr = reinterpret_cast<decltype(cap.cl_ptr)>(lp::stack[env_back + n + 5].get_fun());
			}
			cap.line = 0;
			qlevel = lp::stack[env_back + n + 6].get_fun();
			// std::cerr << cap.cl_i << "\n";
		}
	}

	void Program::try_me_else(int f)
	{
		assert(f != 0);
		int i;
		if (env_i >= env_back) {
			// std::cerr << "try_me_else, eindex (environment)\n";
			if (cap2.line == 0) {
				i = env_i + 3; // place i at beginning of our frame
			} else {
				const instruction& call_instr = *std::prev(cap2); // could also be first try_me_else
				switch (call_instr.opcode) {
				case instruction::CALL:
					i = env_i + call_instr.a3 + 3; // place i at beginning of our frame
					break;
				case instruction::DYN_CALL:
				case instruction::DYN_STACK_CALL:
					i = env_i + call_instr.a2 + 3; // place i at beginning of our frame
					break;
				default: std::cerr << "ERROR: expected CALL or DYN_CALL\n"; assert(false); exit(1);
				}
			}
		} else {
			// std::cerr << "try_me_else, bindex (choice point)\n";
			i = env_back + lp::stack[env_back].arity() + 7;
		}
		const int n = num_of_args;
	    // std::cerr << "Try_me_else, starts at: " << &lp::stack[i] << "\n";
		lp::stack[i] = term(f,n);
		for (int k = 0; k < n; ++k) {
			lp::stack[i+k+1] = term::make_variable(lp::var_regs[k]);
			//std::cerr << "Y" << k << " => " << *lp::var_regs[k] << "\n";
		}
		lp::stack[i+n+1] = term(env_i,heap_i);
		lp::stack[i+n+2] = term(env_back,env_back0);
		lp::stack[i+n+3] = term(tindex,cap2.line);
		//std::cerr << "Try me else, storing tindex: " << tindex << " at stack[" << i+n+3 << "]\n";
		// cap2: store concept ptr, clause index, and code line
		if (cap2.con_ptr) {
			lp::stack[i+n+4] = term(cap2.cl_i,reinterpret_cast<long long>(cap2.con_ptr));
		} else {
			lp::stack[i+n+4] = term(reinterpret_cast<int>(cap2.cl_ptr),reinterpret_cast<long long>(cap2.con_ptr));
		}
		// cap: only need to store concept and clause index=1
		lp::stack[i+n+5] = term(1,reinterpret_cast<long long>(cap.con_ptr)); // jump to index 1 if this fails
		lp::stack[i+n+6] = term(qlevel,0); // store current query level
	    // std::cerr << "  try_me_else, end at: " << &lp::stack[i+n+6] << "\n";
		env_back = i;
		// std::cerr << "  new env_back: " << env_back << "\n";
		heap_back = heap_i;
		++cap;
	}

	void Program::retry_me_else(int m)
	{
		const int f = lp::stack[env_back].get_fun();
		assert(f != 0);
	    // std::cerr << "retry_me_else(" << m << ")\n";
		const int n = lp::stack[env_back].arity();
		for (int k = 0; k < n; ++k) {
			lp::var_regs[k] = lp::stack[env_back + k + 1].get_ptr();
		}
		env_i = lp::stack[env_back + n + 1].get_fun();
		heap_i = lp::stack[env_back + n + 1].arity();
		cap2.line = lp::stack[env_back + n + 3].arity();
		cap2.con_ptr = reinterpret_cast<concept*>(lp::stack[env_back + n + 4].get_int());
		if (cap2.con_ptr) {
			cap2.cl_i = lp::stack[env_back + n + 4].get_fun();
		} else {
			cap2.cl_ptr = reinterpret_cast<decltype(cap2.cl_ptr)>(lp::stack[env_back + n + 4].get_fun());
		}
		lp::stack[env_back + n + 5].set_fun(m); // update index
		unwind_trail(lp::stack[env_back + n + 3].get_fun(), tindex);
		tindex = lp::stack[env_back + n + 3].get_fun();
		heap_back = heap_i;
		++cap;
	}

	void Program::trust_me()
	{
		// std::cerr << "trust_me, env_back: " << env_back << "\n";
		const int n = lp::stack[env_back].arity();
		for (int k = 0; k < n; ++k) {
			//std::cerr << "restoring: X" << k << " => " << *lp::var_regs[k] << " (" << lp::var_regs[k] << "\n";
			lp::var_regs[k] = lp::stack[env_back + k + 1].get_ptr();
		}
		env_i = lp::stack[env_back + n + 1].get_fun();
		heap_i = lp::stack[env_back + n + 1].arity();
		cap2.line = lp::stack[env_back + n + 3].arity();
		cap2.con_ptr = reinterpret_cast<concept*>(lp::stack[env_back + n + 4].get_int());
		if (cap2.con_ptr) {
			cap2.cl_i = lp::stack[env_back + n + 4].get_fun();
		} else {
			cap2.cl_ptr = reinterpret_cast<decltype(cap2.cl_ptr)>(lp::stack[env_back + n + 4].get_fun());
		}
		unwind_trail(lp::stack[env_back + n + 3].get_fun(), tindex);
		tindex = lp::stack[env_back + n + 3].get_fun();
		// Do env_back last since its needed for dereferencing stack
		env_back = lp::stack[env_back + n + 2].get_fun();
		heap_back = heap_i;
		// std::cerr << "  heap_back: " << heap_back << "\n";
		// std::cerr << "  restored env_back: " << env_back << "\n";
		++cap;
	}

	void Program::var(int v)
	{
		term* d = deref(var_regs[v]);
		if (d->is_variable()) {
			++cap;
		} else {
			backtrack();
		}
	}

	void Program::nonvar(int v)
	{
		term* d = deref(var_regs[v]);
		if (d->is_variable()) {
			backtrack();
		} else {
			++cap;
		}
	}

	void Program::atomic(int v)
	{
		term* d = deref(var_regs[v]);
		if (d->is_constant() || d->is_int() || d->is_double()) {
			++cap;
		} else {
			backtrack();
		}
	}

	void Program::atom(int v)
	{
		term* d = deref(var_regs[v]);
		if (d->is_constant()) {
			++cap;
		} else {
			backtrack();
		}
	}

	void Program::number(int v)
	{
		term* d = deref(var_regs[v]);
		if (d->is_int() || d->is_double()) {
			++cap;
		} else {
			backtrack();
		}
	}

	void Program::integer(int v)
	{
		term* d = deref(var_regs[v]);
		if (d->is_int()) {
			++cap;
		} else {
			backtrack();
		}
	}

	void Program::isfloat(int v)
	{
		term* d = deref(var_regs[v]);
		if (d->is_double()) {
			++cap;
		} else {
			backtrack();
		}
	}

	// Unification instruction (=/2)
	void Program::unify_instr(int v, int w)
	{
		//std::cerr << "  unify_instr: " << *var_regs[v] << " <-> " << *var_regs[w] << "\n";
		if (unify(var_regs[v],var_regs[w])) {
			++cap;
		} else {
			backtrack();
		}
	}

	void Program::nonunifiable(int v, int w)
	{
		if (unifiable(var_regs[v],var_regs[w])) {
			backtrack();
		} else {
			++cap;
		}
	}

	void Program::equal(int v, int w)
	{
		try {
			if (rcompare(var_regs[v],var_regs[w]) == 0) {
				++cap;
				return;
			}
		} catch (cyclic_term) { }
		return backtrack();
	}

	void Program::unequal(int v, int w)
	{
		try {
			if (rcompare(var_regs[v],var_regs[w]) != 0) {
				++cap;
				return;
			}
		} catch (cyclic_term) { }
		return backtrack();
	}

	void Program::name(int v, int w)
	{
		term* atom = deref(var_regs[v]);
		term* codelist = deref(var_regs[w]);
		if (codelist->is_variable()) {
			// Make codelist
			std::string s;
			switch (atom->get_type()) {
			case term::CON:
				s = Functor::get_data(atom->get_const()).get_atom();
				if (s.size() >= 2 && s.front() == '\'' && s.back() == '\'') {
					std::string ss = s.substr(1,s.size()-2);
					assert(ss.size()+2 == s.size());
					if (functor_map::is_variable(ss)) {
						s = std::move(ss);
					}
				}
				break;
			case term::INT:
				s = std::to_string(atom->get_int()); break;
			case term::FLT:
				s = std::to_string(atom->get_double()); break;
			case term::VAR:
				DEBUG_WARNING(std::cerr << "Warning: name/2, both arguments are variables\n");
				return backtrack();
			default: 
				DEBUG_WARNING(std::cerr << "Warning: name/2 used with invalid atom: " << *atom << "\n");
				return backtrack();
			}
			assert(!s.empty());
			term* list_begin = &lp::heap[heap_i];
			for (auto c = s.begin(); c != s.end(); ++c) {
				lp::heap[heap_i++] = term::make_list();
				lp::heap[heap_i++] = term::make_int(long long(*c));
				lp::heap[heap_i] = term::make_variable(&lp::heap[heap_i+1]);
				++heap_i;
			}
			lp::heap[heap_i++] = term::make_constant(empty_list_id);
			//std::cerr << "unifying " << *atom << " with " << *list_begin << "\n";
			if (!unify(codelist,list_begin)) return backtrack();
		} else if (codelist->is_list()) {
			// Construct atom from list
			string s;
			term* lptr = codelist;
			while (lptr->is_list()) {
				term* e = deref(lptr+1);
				if (e->is_int()) {
					s += static_cast<char>(e->get_int());
				} else {
					DEBUG_WARNING(std::cerr << "Warning: name/2 used codelist containing " << *e << "\n");
					return backtrack();
				}
				lptr = deref(lptr+2);
			}
			if (functor_map::is_variable(s)) {
				// constant is in variable notation, wrap it around quotes
				s.insert(s.begin(),'\'');
				s += '\'';

			}
			const auto atom_id = fmap.id(s);
			lp::heap[heap_i] = term::make_constant(atom_id);
			// Unify with atom
			if (!unify(atom,&lp::heap[heap_i])) return backtrack();
			++heap_i;
		} else if (codelist->is_constant(empty_list_id)) {
			const auto empty_id = fmap.id("\'\'");
			lp::heap[heap_i] = term::make_constant(empty_id);
			if (!unify(atom,&lp::heap[heap_i])) return backtrack();
			++heap_i;
		} else {
			DEBUG_WARNING(std::cerr << "Warning: name/2 used with invalid codelist: " << *codelist << "\n");
			return backtrack();
		}
		++cap;
	}

	void Program::compare(int x, int y, int z)
	{
		try {
			const auto r = rcompare(var_regs[y],var_regs[z]);
			if (r < 0) {
				heap[heap_i] = term::make_constant(lt_id);
			} else if (r > 0) {
				heap[heap_i] = term::make_constant(gt_id);
			} else {
				heap[heap_i] = term::make_constant(equal_id);
			}
			if (unify(var_regs[x],&heap[heap_i])) {
				++heap_i;
				++cap;
				return;
			}
		} catch (cyclic_term) { }
		return backtrack();
	}

	void Program::before(int v, int w)
	{
		try {
			if (rcompare(var_regs[v],var_regs[w]) < 0) {
				++cap;
				return;
			}
		} catch (cyclic_term) { }
		return backtrack();
	}

	void Program::before_eq(int v, int w)
	{
		try {
			if (rcompare(var_regs[v],var_regs[w]) <= 0) {
				++cap;
				return;
			}
		} catch (cyclic_term) { }
		return backtrack();
	}

	void Program::after(int v, int w)
	{
		try {
			if (rcompare(var_regs[v],var_regs[w]) > 0) {
				++cap;
				return;
			}
		} catch (cyclic_term) { }
		return backtrack();
	}

	void Program::after_eq(int v, int w)
	{
		try {
			if (rcompare(var_regs[v],var_regs[w]) >= 0) {
				++cap;
				return;
			}
		} catch (cyclic_term) { }
		return backtrack();
	}

	void Program::variant(int v, int w)
	{
		try {
			if (is_variant(var_regs[v],var_regs[w])) {
				++cap;
				return;
			} 
		} catch (cyclic_term) { }
		return backtrack();
	}

	void Program::notvariant(int v, int w)
	{
		try {
			if (!is_variant(var_regs[v],var_regs[w])) {
				++cap;
				return;
			} 
		} catch (cyclic_term) { }
		return backtrack();
	}

	void Program::assertz(int v)
	{
		try {
			//std::cerr << "assertz: "<< *deref(var_regs[v]) << "\n";
			std::unique_ptr<Functor> f(term2functor(var_regs[v]));
			//std::cerr << "  assertz term2functor: " << *f << "\n";
			compile_insert(*f,false);
			++cap;
		} catch (cyclic_term) {
			return backtrack();
		}
	}

	void Program::asserta(int v)
	{
		try {
			std::unique_ptr<Functor> f(term2functor(var_regs[v]));
			compile_insert(*f,false,false);
			++cap;
		} catch (cyclic_term) {
			return backtrack();
		}
	}

	void Program::nb_linkarg(int x, int y, int z)
	{
		term* trm = deref(var_regs[y]);
		if (!trm->is_function()) {
			DEBUG_WARNING(std::cerr << "Warning: nb_linkarg/3, expected compound term: " << *trm << "\n");
			return backtrack();
		}
		term* arg = deref(var_regs[x]);
		if (!arg->is_int() || arg->get_int() <= 0 || arg->get_int() > trm->arity()) {
			DEBUG_WARNING(std::cerr << "Warning: nb_linkarg/3, incorrect arg: " << *arg << "\n");
			return backtrack();
		}
		// Overwrite the argument
		term* val = deref(var_regs[z]);
		trm[arg->get_int()] = term::make_variable(val);
		++cap;
	}


	inline int heap_duplicate_term(const term* t, int hi, std::set<const term*>& vars)
	{
		switch (t->get_type()) {
		case term::VAR: case term::EVAR:
			if (t->get_ptr() == t) {
				lp::heap[hi] = term::make_variable(&lp::heap[hi]);
				++hi;
			} else {
				auto p = vars.insert(t);
				if (!p.second) throw cyclic_term();
				hi = heap_duplicate_term(t->get_ptr(),hi,vars);
				vars.erase(t);
			}
			break;
		case term::CON:
			lp::heap[hi++] = term::make_constant(t->get_const());
			break;
		case term::INT:
			lp::heap[hi++] = term::make_int(t->get_int());
			break;
		case term::FLT:
			lp::heap[hi++] = term::make_double(t->get_double());
			break;
		default:
			if (t->is_list()) {
				lp::heap[hi] = term::make_list();
			} else {
				lp::heap[hi] = term::make_function(t->get_fun(),t->arity());
			}
			auto next_hi = hi+t->arity()+1;
			for (int k = 1; k <= t->arity(); ++k) {
				const term* a = t+k;
				lp::heap[hi+k] = term::make_variable(&lp::heap[next_hi]);
				next_hi = heap_duplicate_term(a,next_hi,vars);
			}
			hi = next_hi;
		}
		return hi;
	}

	void Program::duplicate_term(int x, int y)
	{
		try {
			term* t = deref(var_regs[x]);
			std::set<const term*> vars;
			auto new_heap_i = heap_duplicate_term(t,heap_i,vars);
			term* v = deref(var_regs[y]);
			if (unify(v,&lp::heap[heap_i])) {
				heap_i = new_heap_i;
				++cap;
				return;
			}
		} catch (cyclic_term) { }
		return backtrack();
	}

	void Program::retract_all(int v)
	{
		try {
			//std::cerr << "retract: " << v << "\n";
			std::unique_ptr<Functor> t( term2functor(var_regs[v]) );
			//std::cerr << "  retract: " << *t << "\n";
			const sign_t sign = t->signature();
			if (sign.first == false_id && sign.second == 0) {
				// Erase all constraints
				auto at = kb.find(sign_t(if_id,1));
				if (at != kb.end()) kb.erase(at);
			} else {
				auto at = kb.find(sign);
				if (at != kb.end()) {
					for (auto c = at->second.begin(); c != at->second.end(); ) {
						const clause& cl = *c;
						Functor d = decompile(at->first,&cl.front());
						const Functor* dhead = d.head();
						if ( unify(t.get(),d.head()) )  {
							//std::cerr << "Unified, removing\n";
							c = at->second.erase(c);
						} else ++c;	
					}
					// In case we removed it all:
					at = kb.find(t->signature());
					if (at != kb.end()) fix_pointers(at);			
				}
			}
		} catch (cyclic_term) { }
		++cap;
	}

	void Program::define_op(int x, int y, int z)
	{
		const term* prec = deref(var_regs[x]);
		const term* type = deref(var_regs[y]);
		const term* name = deref(var_regs[z]);
		if (!prec->is_int() || !type->is_constant() || !(name->is_constant() || name->is_list())) {
			backtrack();
			return;
		}
		const int p = int(prec->get_int());
		Token::token_type t;
		const std::string str = fmap.get_data(type->get_const()).get_atom();

		if (str == "fx") t = Token::fx;
		else if (str == "fy") t = Token::fy;
		else if (str == "xf") t = Token::xf;
		else if (str == "yf") t = Token::yf;
		else if (str == "xfy") t = Token::xfy;
		else if (str == "yfx") t = Token::yfx;
		else if (str == "xfx") t = Token::xfx;
		else return backtrack();

		if (name->is_constant() && !name->is_constant(empty_list_id)) {
			prolog_dic.add(Token(name->get_const(), t, p));
		} else if (name->is_list()) {
			const term* list = name;
			while (list->is_list()) {
				const term* nm = deref(list+1);
				if (nm->is_constant() && !nm->is_constant(empty_list_id)) {
					prolog_dic.add(Token(nm->get_const(), t, p));
				} else {
					DEBUG_WARNING(std::cerr << "Warning: op/3, invalid name " << fmap.get_data(nm->get_const()) << "\n");
					// continue anyway
				}
				list = deref(list+2);
			}
			// Add last if constant and not []
			if (list->is_constant() && !list->is_constant(empty_list_id)) {
				prolog_dic.add(Token(list->get_const(), t, p));
			}
		} else {
			return backtrack();
		}
		++cap;
	}


	auto Program::fix_pointers(code::iterator c) -> code::iterator
	{
		auto is_ctrl = [](const instruction& i) -> bool { 
			switch (i.opcode) {
			case instruction::NOOP: case instruction::TRY_ME_ELSE: case instruction::RETRY_ME_ELSE: case instruction::TRUST_ME:
				return true;
			default: return false;
			}
		};
		if (c->first == sign_t(if_id,1)) {
			if (c->second.empty()) c = kb.erase(c);
			else ++c;
			// no control negative examples
		} else if (c->second.empty()) {
			c = kb.erase(c);
		} else if (c->second.size() == 1) {
			instruction& i = c->second.front().front();
			if ( is_ctrl(i) ) {
				i.opcode = instruction::NOOP;
			} // else: nothing to do
			++c;
		} else {
			assert(c->second.size() >= 2);
			// at least 2
			// Set first instruction of first clause to TRY_ME_ELSE
			instruction& ia = c->second.front().front();
			if ( is_ctrl(ia) ) {
				ia = instruction(instruction::TRY_ME_ELSE,c->first.first);
			} else {
				c->second.front().insert(c->second.front().begin(),instruction(instruction::TRY_ME_ELSE,c->first.first));
			}
			// First instruction of last clause is TRUST_ME
			instruction& iz = c->second.back().front();
			if ( is_ctrl(iz) ) {
				iz.opcode = instruction::TRUST_ME;
			} else {
				c->second.back().insert(c->second.back().begin(),instruction(instruction::TRUST_ME));
			}
			// Adjust intermediate clauses
			auto last = --c->second.end();
			int k = 2; // first RETRY_ME_ELSE is index 1 and links to index 2
			for (auto j = ++c->second.begin(); j != last; ++j,++k) {
				instruction& i = j->front();
				if ( is_ctrl(i) ) {
					i = instruction(instruction::RETRY_ME_ELSE,k);
				} else {
					j->insert(j->begin(),instruction(instruction::RETRY_ME_ELSE,k));
				}
			}
			// Move on to next c
			++c;
		}
		return c;
	}


	void Program::listing(bool dcomp)
	{
		// Print all mode declarations
		if (!modev.empty()) {
			std::cout << "Mode Declarations:\n";
			for (auto m = modev.begin(); m != modev.end(); ++m) {
				std::cout << "  " << *m << "\n";
			}
			std::cout << "\n";
		}

		auto is_neg = [&](const sign_t& s){ return s.first == if_id && s.second == 1; };
		// Print everything in code_map
		for (auto i = kb.begin(); i != kb.end(); ++i) {
			const concept& conref = i->second;
			int defc = 1;
			std::cout << fmap.get_data(i->first.first) << "/" << i->first.second << ":\n";
			for (auto cc = conref.begin(); cc != conref.end(); ++cc,++defc) {
				if (dcomp) {
					Functor t = (is_neg(i->first) ? decompile(&cc->front()) : decompile(i->first,&cc->front()));
					std::cout << " " << defc << ": " << t << ".\n";
				} else {
					std::cout << " Definition " << defc << ":\n";
					for (auto i = cc->begin(); i != cc->end(); ++i) {
						std::cout << "  " << *i << "\n";
					}
				}
			}
		}
		std::cout << "\n";
		++cap;
	}

	void Program::listing(int v, bool dcomp)
	{
		int defc = 0;
		const term* t = deref(var_regs[v]);
		if (t->is_constant(modeh_id)) {
			// Print modeh
			for (auto& m : modev) {
				if (m.is_head()) {
					std::cout << " " << ++defc << ":  " << m << "\n";
				}
			}
		} else if (t->is_constant(modeb_id)) {
			// Print modeb
			for (auto& m : modev) {
				if (!m.is_head()) {
					std::cout << " " << ++defc << ":  " << m << "\n";
				}
			}
		} else if (t->is_function(div_id,2) && deref(t+1)->is_constant() && deref(t+2)->is_int()) {
			// Match signature
			const sign_t s(deref(t+1)->get_const(),static_cast<int>(deref(t+2)->get_int()));
			const bool is_neg = (s.first == if_id && s.second == 1);
			auto at = kb.find(s);
			if (at != kb.end()) {
				for (auto cl = at->second.begin(); cl != at->second.end(); ++cl) {
					if (dcomp) {
						Functor t = (is_neg ? decompile(*cl) : decompile(s,*cl));
						std::cout << " " << ++defc << ": " << t << ".\n";
					} else {
						std::cout << " Definition " << ++defc << ":\n";
						for (auto& instr : *cl) std::cout << instr << "\n";
					}
				}
				std::cout << "\n";
			}
		} else {
			// Unify
			try {
				unique_ptr<Functor> fun;
				fun.reset( term2functor(t) );
				const sign_t s = t->signature();
				const bool is_neg = (s.first == if_id && s.second == 1);
				auto at = kb.find(s);
				if (at != kb.end()) {
					for (auto cl = at->second.begin(); cl != at->second.end(); ++cl) {
						const Functor t = (is_neg ? decompile(*cl) : decompile(s,*cl));
						if (!unify(fun.get(),&t)) continue;
						if (dcomp) {
							std::cout << " " << ++defc << ": " << t << ".\n";
						} else {
							std::cout << " Definition " << ++defc << ":\n";
							for (auto& instr : *cl) std::cout << instr << "\n";
						}
					}
					std::cout << "\n";
				}
			} catch (cyclic_term) {
				return backtrack();
			}
		}

		if (defc > 0) {
			++cap;
		} else {
			return backtrack();
		}
	}


	void Program::write(int v)
	{
		print_infix(std::cout,deref(var_regs[v]));
		++cap;
	}

	void Program::nl()
	{
		std::cout << "\n";
		++cap;
	}

	bool consult_file(const boost::filesystem::path& p, Program& kb, bool fail_free)
	{
		// Update current working directory
		auto old_cwd = CURRENT_PATH;
		try {
			//std::cerr << "current path: " << CURRENT_PATH << "\n";
			//std::cerr << "requested path: " << p << "\n";
			if (p.is_absolute()) {
				CURRENT_PATH = p.parent_path();
				//std::cerr << "absolute path: " << CURRENT_PATH << "\n";
			} else {
				if (boost::filesystem::is_regular_file(CURRENT_PATH)) CURRENT_PATH.remove_filename();
				//std::cerr << "CURRENT_PATH after removing filename: " << CURRENT_PATH << "\n";
				CURRENT_PATH /= p;
				//std::cerr << "relative path: " << CURRENT_PATH << "\n";
			}
			//std::cerr << "Trying to open: " << CURRENT_PATH.string() << "\n";
			std::ifstream inf(CURRENT_PATH.string());
			if (!inf.is_open()) {
				// Try adding .pl
				if (CURRENT_PATH.has_filename()) {
					const std::string old_path = CURRENT_PATH.string();
					const std::string new_filename = CURRENT_PATH.filename().string() + ".pl";
					CURRENT_PATH.remove_filename();
					CURRENT_PATH /= new_filename;
					inf.open(CURRENT_PATH.string());
					if (!inf.is_open()) {
						// Restore CURRENT_PATH
						CURRENT_PATH = std::move(old_cwd);
						// Check require_files
						if (!fail_free && kb.params.is_set(parameters::require_files)) {
							std::cerr << "Error: could not open " << old_path << "\n";
							throw destruct_and_exit();
						} else {
							DEBUG_WARNING( std::cerr << "Warning: could not open " << old_path << "\n" );
							return false;
						}
					}
				}
			}
			// Read file
			DEBUG_INFO( std::cerr << "Opened file " << CURRENT_PATH << "\n" );
			inf >> kb;
		} catch (const lp::parse_error& pe) {
			DEBUG_WARNING(std::cerr << "Error: while consulting file " << CURRENT_PATH << ", " << pe.what() << "\n");
			CURRENT_PATH = std::move(old_cwd);
			throw;
		} catch (...) {
			CURRENT_PATH = std::move(old_cwd);
			throw;
		}
		// Restore CURRENT_PATH
		CURRENT_PATH = std::move(old_cwd);
		return true;
	}

	inline bool consult_file(const term* t, Program& kb)
	{
		// Open file
		if (!t->is_constant()) {
			DEBUG_WARNING( std::cerr << "Warning: invalid filename " << *t << "\n" );
			return false;
		}
		boost::filesystem::path p = lp::fmap.get_data(t->get_const()).get_atom();
		return consult_file(p,kb);
	}

	void Program::consult_list(int v, int w)
	{
		try {
			const term* elem = deref(var_regs[v]);
			const term* ptr = deref(var_regs[w]);
			consult_file(elem,*this); // read first
			std::set<const term*> vars; // cycle proof
			for ( ; ptr->is_list(); ptr = deref_nocycle(ptr+2,vars)) {
				consult_file(deref(ptr+1),*this);
			}
			// Read last list element
			if (!ptr->is_constant(empty_list_id) ) {
				consult_file(ptr,*this);
			}
			++cap;
			return;
		} catch (cyclic_term) {
		} catch (lp::parse_error) { }
		return backtrack();
	}

	void Program::consult(int v)
	{
		const term* addr = deref(var_regs[v]);
		//std::cerr << "ADDR: " << *addr << "\n";
		try {
			if (!consult_file(addr,*this)) {
				backtrack();
				return;
			}
		} catch (lp::parse_error) {
			return backtrack();
		}
		++cap;
		//std::cerr << "instruction after consult: " << *cap << "\n";
	}

	void Program::is(int v, int w)
	{
		bool fail = false;
		term ans;
		try {
			ans = eval(var_regs[w]);
		} catch (useful::non_numeric) {
			backtrack();
			return;
		}
		term* addr = deref(var_regs[v]);
		switch (addr->get_type()) {
		case term::VAR: case term::EVAR:
			if (ans.is_int()) lp::heap[heap_i] = term(ans.get_int());
			else lp::heap[heap_i] = term(ans.get_double());
			bind(addr,&lp::heap[heap_i]);
			++heap_i;
			break;
		case term::INT:
			fail = !ans.is_int() || addr->get_int() != ans.get_int();
			break;
		case term::FLT:
			fail = !ans.is_double() || addr->get_double() != ans.get_double();
			break;
		default:
			fail = true;
		}

		if (fail) {
			backtrack();
		} else {
			++cap;
		}
	}

	void Program::numequal(int v, int w)
	{
		try {
			if (num_compare(var_regs[v],var_regs[w]) == 0) {
				++cap;
				return;
			}
		} catch (useful::non_numeric) {}
		backtrack();
	}

	void Program::numunequal(int v, int w)
	{
		try {
			if (num_compare(var_regs[v],var_regs[w]) != 0) {
				++cap;
				return;
			}
		} catch (useful::non_numeric) {}
		backtrack();
	}

	void Program::numless(int v, int w)
	{
		try {
			if (num_compare(var_regs[v],var_regs[w]) < 0) {
				++cap;
				return;
			}
		} catch (useful::non_numeric) {}
		backtrack();
	}

	void Program::numlesseq(int v, int w)
	{
		try {
			//std::cerr << "=</2 : " << *deref(var_regs[v]) << " =< " << *deref(var_regs[w]) << "\n";
			if (num_compare(var_regs[v],var_regs[w]) <= 0) {
				++cap;
				return;
			}
		} catch (useful::non_numeric) {}
		backtrack();
	}

	void Program::numgreater(int v, int w)
	{
		try {
			if (num_compare(var_regs[v],var_regs[w]) > 0) {
				++cap;
				return;
			}
		} catch (useful::non_numeric) {}
		backtrack();
	}

	void Program::numgreatereq(int v, int w)
	{
		try {
			if (num_compare(var_regs[v],var_regs[w]) >= 0) {
				++cap;
				return;
			}
		} catch (useful::non_numeric) {}
		backtrack();
	}

	void Program::univ(int v, int w)
	{
		term* t1 = deref(var_regs[v]);
		term* t2 = deref(var_regs[w]);
		if (t1->is_variable()) {
			// Build functor from list
			// if (t2->is_variable()) return backtrack();
			if (!t2->is_list()) {
				DEBUG_WARNING( std::cerr << "Warning: second argument to univ must be list\n" );
				return backtrack();
			}
			// Store all arguments in array
			term* ptr = t2;
			std::vector<term*> argv;
			term* func = deref(ptr+1);
			if (!func->is_constant()) {
				DEBUG_WARNING( std::cerr << "Warning: univ expects constant as head, read " << *t2 << "\n" );
				return backtrack();
			}
			ptr = ptr+2;
			std::set<const term*> vars;
			for (;;) {
				ptr = deref_nocycle(ptr,vars);
				if (!ptr) {
					DEBUG_WARNING(std::cerr << "Warning: cyclic term in univ\n");
					return backtrack(); 
				}
				if (ptr->is_constant(lp::empty_list_id)) {
					break;
				} else if (ptr->is_list()) {
					argv.push_back( ptr+1 );
					ptr = ptr + 2;
				} else {
					return backtrack();
				}
			}
			// Make functor and bind it to t1
			if (argv.empty()) {
				lp::heap[heap_i] = term::make_constant(func->arity());
			} else {
				lp::heap[heap_i] = term::make_function(func->arity(),argv.size());
			}
			// Bind t1 to functor
			bind(t1,&lp::heap[heap_i]);
			++heap_i;
			// Create arguments
			for (auto i = argv.begin(); i != argv.end(); ++i) {
				lp::heap[heap_i++] = term::make_variable(*i);
			}
		} else if (t2->is_variable()) {
			// Build list from functor
			term* farg;
			int argc;
			if (t1->is_constant()) {
				lp::heap[heap_i] = term::make_list();
				bind(t2,&lp::heap[heap_i]);
				++heap_i;
				lp::heap[heap_i++] = term::make_constant(t1->arity());
				argc = 0;
			} else if (t1->is_function()) {
				lp::heap[heap_i] = term::make_list();
				bind(t2,&lp::heap[heap_i]);
				++heap_i;
				lp::heap[heap_i++] = term::make_constant(t1->get_type());
				argc = t1->arity();
				farg = t1+1;
			} else {
				return backtrack();
			}
			// Put references to functor arguments in list
			for (int k = 0; ; ++k,++farg) {
				if (k >= argc) {
					// Finish
					lp::heap[heap_i++] = term::make_constant(lp::empty_list_id);
					break;
				} else {
					// Make reference to functor arg
					lp::heap[heap_i] = term::make_variable(&lp::heap[heap_i+1]);
					++heap_i;
					lp::heap[heap_i++] = term::make_list();
					lp::heap[heap_i++] = term::make_variable(farg);
				}
			}
		} else {
			// A1 must be functor, A2 list
			if (!t2->is_list()) {
				DEBUG_WARNING( std::cerr << "Warning: second argument to univ must be list\n" );
				return backtrack();
			}
			// Traverse functor and list, unifying
			// Special case: head of list must be constant corresponding to t1's predicate
			if (t1->is_constant()) {
				if (!unify(t1,t2+1)) return backtrack();
				lp::heap[heap_i] = term::make_constant(lp::empty_list_id);
				if (!unify(t2+2,&lp::heap[heap_i])) return backtrack();
				++heap_i;
			} else if (t1->is_function()) {
				lp::heap[heap_i] = term::make_constant(t1->get_type());
				if (!unify(t2+1,&lp::heap[heap_i])) return backtrack();
				++heap_i;
				term* lptr = t2;
				for (int k = 1; k <= t1->arity(); ++k) {
					lptr = deref(lptr+2);
					if (!lptr->is_list()) return backtrack();
					term* arg = deref(t1+k);
					if (!unify(arg,lptr+1)) return backtrack();
				}
				// Avoid using heap for last [] if possible
				lptr = deref(lptr+2);
				switch (lptr->get_type()) {
				case term::VAR: case term::EVAR:
					lp::heap[heap_i] = term::make_constant(lp::empty_list_id);
					bind(lptr,&lp::heap[heap_i]);
					++heap_i;
					break;
				case term::CON:
					if (lptr->arity() == lp::empty_list_id) {
						break;
					}
					// else: fall through
				default:
					return backtrack();
				}
			} else {
				DEBUG_WARNING( std::cerr << "Warning: first argument to univ must be functor\n" );
				return backtrack();
			}
		}
		++cap;
	}

	void Program::functor(int x, int y, int z)
	{
		term* t = deref(var_regs[x]);
		term* name = deref(var_regs[y]);
		term* ar = deref(var_regs[z]);
		if (t->is_variable()) {
			if (!ar->is_int()) return backtrack();
			if (ar->get_int() == 0) {
				if (!name->is_constant() && !name->is_int() && !name->is_double()) return backtrack();
				lp::heap[heap_i] = *name;
				if (!unify(t,&lp::heap[heap_i])) return backtrack();
				++heap_i;
			} else {
				if (!name->is_constant()) return backtrack();
				lp::heap[heap_i] = term(name->arity(),static_cast<int>(ar->get_int()));
				bind(t,&lp::heap[heap_i]);
				++heap_i;
				for (int k = 1; k <= ar->get_int(); ++k) {
					lp::heap[heap_i] = term::make_variable(&lp::heap[heap_i]);
					++heap_i;
				}
			}
		} else {
			switch (t->get_type()) {
			case term::CON:
			case term::INT:
			case term::FLT:
				if (!unify(name,t)) return backtrack();
				lp::heap[heap_i] = term::make_int(0LL);
				if (!unify(ar,&lp::heap[heap_i])) return backtrack();
				++heap_i;
				break;
			default:
				assert( t->is_function() );
				lp::heap[heap_i] = term::make_constant(t->get_type());
				if (!unify(name,&lp::heap[heap_i])) return backtrack();
				++heap_i;
				lp::heap[heap_i] = term(static_cast<long long>(t->arity()));
				if (!unify(ar,&lp::heap[heap_i])) return backtrack();
				++heap_i;
				break;
			}
		}
		++cap;
	}

	void Program::arg(int x, int y, int z)
	{
		term* t = deref(var_regs[y]);
		if (!t->is_function()) return backtrack();
		term* argt = deref(var_regs[x]);
		term* val = deref(var_regs[z]);
		if (argt->is_variable()) {
			// Generate arguments during backtracking
			// TODO (non-ISO)
			return backtrack();
		} else if (argt->is_int()) {
			const int argk = static_cast<int>(argt->get_int());
			if (argk < 0 || argk >= t->arity()) return backtrack();
			if (!unify(val,t+argk)) return backtrack();
		} else {
			return backtrack();
		}
		++cap;
	}

	void Program::neck_cut()
	{
		if (env_back > env_back0) {
			tidy_trail();
			env_back = env_back0;
		}
		++cap;
	}

	void Program::get_level(int yi)
	{
	//std::cerr << "get_level, env_i: " << env_i << ", storing env_back0: " << env_back0 << "\n";
		lp::stack[env_i + 1 + yi] = term(0,env_back0);
	    //std::cerr << "  get_level, storage at: " << &lp::stack[env_i + 1 + yi] << "\n";
	    //std::cerr << "  get_level, stored env_back: " << lp::stack[env_i + 1 + yi].arity() << "\n";
		++cap;
	}

	void Program::cut(int yi)
	{
		//assert(lp::stack[env_i + 2 + yi].is_nil());
	    //std::cerr << "cut, env_back: " << env_back << "\n";
		if (env_back > lp::stack[env_i + 1 + yi].arity()) {
		    //std::cerr << "  cut, before tidy, stack val should be 0: " << lp::stack[env_i + 1 + yi].get_type() << "\n";
		    //std::cerr << "  cut, before tidy, stack env_back: " << lp::stack[env_i + 1 + yi].arity() << "\n";
			tidy_trail();
		    //std::cerr << "  cut, after  tidy, stack env_back: " << lp::stack[env_i + 1 + yi].arity() << "\n";
			env_back = lp::stack[env_i + 1 + yi].arity();
		    //std::cerr << "  cut, storage at: " << &lp::stack[env_i + 1 + yi] << "\n";
		    //std::cerr << "  cut, updated env_back: " << env_back << "\n";
		}
		++cap;
	}

	void Program::determination(int v, int w, bool is_include)
	{
		const term* target = deref(var_regs[v]);
		auto is_sign = [](const term* s){ return s->is_function(div_id,2) && s[1].is_constant() && s[2].is_int() && s[2].get_int() >= 0; };
		if (!is_sign(target)) {
			DEBUG_CRUCIAL(std::cerr << "Error: determination, target predicate must be p/n, not " << *target << "\n");
			return backtrack();
		}
		const term* body_lit = deref(var_regs[w]);
		if (!is_sign(body_lit)) {
			DEBUG_CRUCIAL(std::cerr << "Error: determination, body predicate must be p/n, not " << *body_lit << "\n");
			return backtrack();
		}
		const auto p = std::make_pair(
			sign_t(target[1].get_const(),int(target[2].get_int())),
			sign_t(body_lit[1].get_const(),int(body_lit[2].get_int())));
		if (is_include) {
			includes.insert(std::move(p));
		} else {
			excludes.insert(std::move(p));
		}
		++cap;
	}

	void Program::modeh(int v, int w)
	{
		try {
			// Set Recall
			int recall = std::numeric_limits<int>::max();
			const term* rec = deref(var_regs[v]);
			if (rec->is_int()) {
				recall = static_cast<int>(rec->get_int());
			}
			// Parse
			std::unique_ptr<Functor> f( term2functor(var_regs[w]) );
			const Mode m(*f,true,recall);
			// Do not create duplicate modes
			if (std::find(modev.begin(),modev.end(),m) == modev.end()) {
				modev.push_back(std::move(m));
			} else {
				DEBUG_WARNING(std::cerr << "Warning: ignoring duplicate modeh " << m << "\n");
			}
			++cap;
		} catch (cyclic_term) {
			DEBUG_CRUCIAL(std::cerr << "Error: cyclic term used in modeh goal\n");
			return backtrack();
		}
	}

	void Program::modeb(int v, int w, int s)
	{
		try {
			// std::cerr << "wam modeb: " << v << ", " << w << "\n";
			// std::cerr << "recall: " << *deref(var_regs[v]) << "\n";
			// Set Recall
			int recall = std::numeric_limits<int>::max();
			const term* rec = deref(var_regs[v]);
			if (rec->is_int()) {
				recall = static_cast<int>(rec->get_int());
			}
			// Parse list options
			int moc = std::numeric_limits<int>::max(); // max occurs
			bool functional = false;
			int reflexive = 0;
			int symmetric = 0;
			bool transitive = 0;
			bool idempotent = 0;
			const term* li = (s >= 0 ? deref(var_regs[s]) : nullptr);
			if (li) {
				auto parse = [&](const term* e){
					if (e->is_constant()) {
						switch (e->get_const()) {
						case functional_id: functional = true; break;
						case reflexive_id: reflexive = 1; break;
						case symmetric_id: symmetric = 1; break;
						case irreflexive_id: reflexive = -1; break;
						case antisymmetric_id: symmetric = -1; break;
						case transitive_id: transitive = 1; break;
						case idempotent_id: idempotent = 1; break;
						default:
							DEBUG_WARNING(std::cerr << "Warning: ignoring unknown modeb flag " << *e << "\n");
						}
					} else if (e->is_int()) {
						moc = static_cast<int>(e->get_int());
					} else {
						DEBUG_WARNING(std::cerr << "Warning: ignoring unknown modeb flag " << *e << "\n");
					}
				};
				for ( ; li->is_list(); li = deref(li+2)) {
					parse(deref(li+1));
				}
				if (!li->is_constant(empty_list_id)) parse(li);
			}
			// Parse
			// std::cerr << "declaration: " << *deref(var_regs[w]) << "\n";
			std::unique_ptr<Functor> f( term2functor(var_regs[w]) );
			Mode mode(*f,false,recall);
			mode.props[Mode::maxc] = moc;
			mode.props[Mode::functional] = functional;
			mode.props[Mode::reflexive] = reflexive;
			mode.props[Mode::symmetric] = symmetric;
			mode.props[Mode::transitive] = transitive;
			mode.props[Mode::idempotent] = idempotent;
			// Do not create duplicate modes
			if (std::find(modev.begin(),modev.end(),mode) == modev.end()) {
				modev.push_back(std::move(mode));
			} else {
				DEBUG_WARNING(std::cerr << "Warning: ignoring duplicate modeb " << mode << "\n");
			}
			++cap;
		} catch (cyclic_term) {
			DEBUG_CRUCIAL(std::cerr << "Error: cyclic term used in modeb goal\n");
			return backtrack();
		}
	}

	void Program::add_constraint(int v, int w, bool is_bottom)
	{
		implication ip;
		try {
			// Add LHS
			const term* ptr = nullptr;
			std::map<const term*,id_type> vmap;
			if (v >= 0) {
				ptr = deref(var_regs[v]);
				auto add_literal = [&](const term* t){
					std::unique_ptr<Functor> f( term2functor(t,vmap) );
					ip.ante.push_back(std::move(*f.release()));
				};
				for ( ; ptr->is_function(sequence_id,2); ptr = deref(ptr+2)) {
					add_literal(deref(ptr+1));
				}
				add_literal(ptr);
			} // else: don't add any literals
			// Add RHS
			std::unique_ptr<Functor> f(term2functor(deref(var_regs[w]),vmap));
			ip.cons = std::move(*f.release());
		} catch (cyclic_term) {
			DEBUG_CRUCIAL(std::cerr << "Warning: cyclic term while adding constraint\n");
			return backtrack();
		}
		if (is_bottom) {
			block.push_back(std::move(ip));
		} else {
			imp.push_back(std::move(ip));
		}
		++cap;
	}

	void Program::set_param(int v, int w)
	{
		bool fail = false;
		const term* name = deref(var_regs[v]);
		if (name->is_constant()) {
			const data& dat = fmap.get_data(name->get_const());
			if (dat.is_atom()) {
				try {
					data& opt = params[dat.get_atom()];
					const term* value = deref(var_regs[w]);
					switch (value->get_type()) {
					case term::CON:
						opt = data(fmap.get_atom(value->get_const())); 
						break;
					case term::INT:
						opt = value->get_int();
						break;
					case term::FLT:
						opt = value->get_double();
						break;
					default:
						fail = true;
					}
				} catch (invalid_name) {
					fail = true;
				}
			} else {
				fail = true;
			}
		} else {
			fail = true;
		}
		if (fail) {
			backtrack();
		} else {
			++cap;
		}
	}

	void Program::get_param(int v, int w)
	{
		bool fail = false;
		const term* name = deref(var_regs[v]);
	    //std::cerr << "name: " << *name << "\n";
		if (name->is_constant()) {
		    //std::cerr << "name is constant\n";
			const data& dat = fmap.get_data(name->get_const());
			if (dat.is_atom()) {
			    //std::cerr << "dat is atom\n";
				try {
					const data& opt = params[dat.get_atom()];
				    //std::cerr << "opt " << dat.get_atom() << " is valid\n";
					term* var = deref(var_regs[w]);
				    //std::cerr << "var: " << *var << "\n";
					switch (opt.type()) {
					case data::int_tag:
						if (var->is_variable()) {
							lp::heap[heap_i] = term::make_int(opt.get_int());
							bind(var,&lp::heap[heap_i]);
							++heap_i;
						} else if (!var->is_int() || var->get_int() != opt.get_int()) {
							fail = true;
						}
						break;
					case data::float_tag:
						if (var->is_variable()) {
							lp::heap[heap_i] = term::make_double(opt.get_float());
							bind(var,&lp::heap[heap_i]);
							++heap_i;
						} else if (!var->is_double() || var->get_double() != opt.get_float()) {
							fail = true;
						}
						break;
					default:
						assert( opt.is_atom() );
						if (var->is_variable()) {
							lp::heap[heap_i] = term::make_constant(fmap.id(opt.get_atom()));
							bind(var,&lp::heap[heap_i]);
							++heap_i;
						} else if (!var->is_constant() || var->get_const() != fmap.id(opt.get_atom())) {
							fail = true;
						}
						break;
					}
				} catch (invalid_name) {
					fail = true;
				}
			} else {
				fail = true;
			}
		} else {
			fail = true;
		}
		if (fail) {
			backtrack();
		} else {
			++cap;
		}
	}

	void Program::generalize()
	{
		try {
			this->generalize("nrsample");
			++cap;
		} catch (induction_exception) {
			backtrack();
		}
	}

	void Program::generalize(int v)
	{
		term* t = deref(var_regs[v]);
		if (!t->is_constant()) {
			backtrack();
			return;
		} else {
			try {
				switch (t->get_const()) {
				case nrsample_id:
					this->generalize("nrsample");
					break;
				case id_nrsample_id:
					this->generalize("id_nrsample");
					break;
				case emulate_nrsample_id:
					this->generalize("emulate_nrsample");
					break;
				case enumerate_id:
					this->generalize("enumerate");
					break;
				case heuristic_id:
					this->generalize("heuristic");
					break;
				default:
					backtrack();
					return;
				}
				++cap;
			} catch (induction_exception) {
				return backtrack();
			}
		}
	}


	std::ostream& operator<<(std::ostream& os, const instruction& i)
	{
		switch (i.opcode) {
		case instruction::STOP: os << "STOP"; return os;
		case instruction::NOOP: os << "NOOP"; ; break;
		case instruction::PROCEED: os << "PROCEED"; ; break;
		case instruction::CALL: os << "CALL " << lp::fmap.get_data(i.a1) << "/" << i.a2 << ", " << i.a3; break;
		case instruction::EXEC: os << "EXEC " << lp::fmap.get_data(i.a1) << "/" << i.a2; break;
		case instruction::DYN_CALL: os << "DYN_CALL X" << i.a1 << ", " << i.a2; break;
		case instruction::DYN_EXEC: os << "DYN_EXEC X" << i.a1; break;
		case instruction::DYN_STACK_CALL: os << "DYN_CALL Y" << i.a1 << ", " << i.a2; break;
		case instruction::DYN_STACK_EXEC: os << "DYN_EXEC Y" << i.a1; break;
		case instruction::PUT_STR: os << "PUT_STR " << lp::fmap.get_data(i.a1) << "/" << i.a2 << ", X" << i.a3; break;
		case instruction::PUT_LIS: os << "PUT_LIS X" << i.a1; break;
		case instruction::SET_VAR: os << "SET_VAR X" << i.a1; break;
		case instruction::SET_VARY: os << "SET_VAR Y" << i.a1; break;
		case instruction::SET_VAL: os << "SET_VAL X" << i.a1; break;
		case instruction::SET_VALY: os << "SET_VAL Y" << i.a1; break;
		case instruction::GET_STR: 
			os << "GET_STR " << lp::fmap.get_data(i.a1) << "/" << i.a2 << ", X" << i.a3;
			break;
		case instruction::GET_LIS: os << "GET_LIS X" << i.a1; break; 
		case instruction::UNI_VAR: os << "UNI_VAR X" << i.a1; break;
		case instruction::UNI_VARY: os << "UNI_VAR Y" << i.a1; break;
		case instruction::UNI_VAL: os << "UNI_VAL X" << i.a1; break;
		case instruction::UNI_VALY: os << "UNI_VAL Y" << i.a1; break;
		case instruction::PUT_VAR: os << "PUT_VAR X" << i.a1 << ", X" << i.a2; break;
		case instruction::PUT_VAL: os << "PUT_VAL X" << i.a1 << ", X" << i.a2; break;
		case instruction::GET_VAR: os << "GET_VAR X" << i.a1 << ", X" << i.a2; break;
		case instruction::GET_VAL: os << "GET_VAL X" << i.a1 << ", X" << i.a2; break;
		case instruction::GET_VALY: os << "GET_VAL Y" << i.a1 << ", X" << i.a2; break;
		case instruction::ALLOC: os << "ALLOC"; break;
		case instruction::DEALLOC: os << "DEALLOC"; break;
		case instruction::GLOBALIZE_SVARS: os << "GLOBALIZE_SVARS"; break;
		case instruction::GET_VARY: os << "GET_VAR Y" << i.a1 << ", X" << i.a2; break;
		case instruction::PUT_VARY: os << "PUT_VAR Y" << i.a1 << ", X" << i.a2; break;
		case instruction::PUT_VALY: os << "PUT_VAL Y" << i.a1 << ", X" << i.a2; break;
		case instruction::TRY_ME_ELSE: os << "TRY_ME_ELSE " << Functor::get_data(i.a1); break;
		case instruction::RETRY_ME_ELSE: os << "RETRY_ME_ELSE " << i.a1; break;
		case instruction::TRUST_ME: os << "TRUST_ME"; break;
		case instruction::PROMPT: os << "PROMPT"; return os;
			//case instruction::CONTINUE: wam_continue(); return os;
		case instruction::PUT_CON: os << "PUT_CON " << fmap.get_data(i.a1) << ", X" << i.a2; break;
		case instruction::GET_CON: os << "GET_CON " << fmap.get_data(i.a1) << ", X" << i.a2; break;
		case instruction::SET_CON: os << "SET_CON " << fmap.get_data(i.a1); break;
		case instruction::UNI_CON: os << "UNI_CON " << fmap.get_data(i.a1); break;
		case instruction::PUT_INT: os << "PUT_INT " << i.i1 << ", X" << i.a2; break;
		case instruction::GET_INT: os << "GET_INT " << i.i1 << ", X" << i.a2; break;
		case instruction::SET_INT: os << "SET_INT " << i.i1; break;
		case instruction::UNI_INT: os << "UNI_INT " << i.i1; break;
		case instruction::PUT_FLT: os << "PUT_FLT " << i.d1 << ", X" << i.a2; break;
		case instruction::GET_FLT: os << "GET_FLT " << i.d1 << ", X" << i.a2; break;
		case instruction::SET_FLT: os << "SET_FLT " << i.d1; break;
		case instruction::UNI_FLT: os << "UNI_FLT " << i.d1; break;
		//case instruction::PUT_LIS: os << "PUT_LIS X" << i.a1; break;
		//case instruction::GET_LIS: os << "GET_LIS X" << i.a1; break;
		case instruction::SET_VOID: os << "SET_VOID " << i.a1; break;
		case instruction::UNI_VOID: os << "UNI_VOID " << i.a1; break;
		case instruction::UNIFY: os << "UNIFY " << i.a1 << ", " << i.a2; break;
		case instruction::PUT_UNSAFE_VALY: os << "PUT_UNSAFE_VALY Y" << i.a1 << ", X" << i.a2; break;
		case instruction::SET_LOCAL_VAL: os << "SET_LOCAL_VAL X" << i.a1; break;
		case instruction::SET_LOCAL_VALY: os << "SET_LOCAL_VAL Y" << i.a1; break;
		case instruction::UNI_LOCAL_VAL: os << "UNI_LOCAL_VAL X" << i.a1; break;
		case instruction::REG_HVAR: os << "REG_HVAR X" << Functor::get_data(i.a1); break;
		case instruction::REG_SVAR: os << "REG_SVAR " << Functor::get_data(i.a1) << ", Y" << i.a2; break;
		case instruction::REG_ARG: os << "REG_ARG X" << i.a1; break;
		case instruction::NECK_CUT: os << "NECK_CUT"; break;
		case instruction::GET_LVL: os << "GET_LVL Y" << i.a1; break;
		case instruction::CUT: os << "CUT Y" << i.a1; break;
		case instruction::FAIL: os << "FAIL"; break;
		case instruction::VAR: os << "VAR X" << i.a1; break;
		case instruction::NONVAR: os << "NONVAR X" << i.a1; break;
		case instruction::ATOMIC: os << "ATOMIC X" << i.a1; break;
		//case instruction::DYNCALL: os << "DYNCALL " << i.a1 << ", " << i.a2; break;
		case instruction::ATOM: os << "ATOM X" << i.a1; break;
		case instruction::NUMBER: os << "NUMBER X" << i.a1; break;
		case instruction::INTEGER: os << "INTEGER X" << i.a1; break;
		case instruction::FLOAT: os << "FLOAT X" << i.a1; break;
		case instruction::NONUNIFIABLE: os << "nonunifiable X" << i.a1 << ", X" << i.a2; break;
		case instruction::EQUAL: os << "equal X" << i.a1 << ", X" << i.a2; break;
		case instruction::UNEQUAL: os << "unequal X" << i.a1 << ", X" << i.a2; break;
		case instruction::BEFORE: os << "before X" << i.a1 << ", X" << i.a2; break;
		case instruction::BEFORE_EQ: os << "before_eq X" << i.a1 << ", X" << i.a2; break;
		case instruction::AFTER: os << "after X" << i.a1 << ", X" << i.a2; break;
		case instruction::AFTER_EQ: os << "after_eq X" << i.a1 << ", X" << i.a2; break;
		case instruction::COMPARE: os << "compare X" << i.a1 << ", X" << i.a2 << ", Z" << i.a3; break;
		case instruction::VARIANT: os << "variant X" << i.a1 << ", X" << i.a2; break;
		case instruction::NOT_VARIANT: os << "not_variant X" << i.a1 << ", X" << i.a2; break;
		case instruction::HALT: os << "HALT"; break;
		case instruction::ASSERT: os << "ASSERT X" << i.a1; break;
		case instruction::ASSERTA: os << "ASSERTA X" << i.a1; break;
		case instruction::NB_LINKARG: os << "NB_LINKARG X" << i.a1 << ", X" << i.a2 << ", X" << i.a3; break;
		case instruction::DUPE_TERM: os << "DUPE_TERM X" << i.a1 << ", X" << i.a2; break;
		case instruction::RETRACTALL: os << "RETRACTALL X" << i.a1; break;
		case instruction::OP: os << "op X" << i.a1 << ", X" << i.a2 << ", X" << i.a3; break;
		case instruction::LISTING: os << "LISTING"; break;
		case instruction::LISTING_WAM: os << "LISTING_WAM"; break;
		case instruction::LISTING1: os << "LISTING1"; break;
		case instruction::LISTING_WAM1: os << "LISTING_WAM1"; break;
		case instruction::WRITE: os << "WRITE " << i.a1; break;
		case instruction::NL: os << "NL "; break;
		case instruction::CONSULT: os << "CONSULT " << i.a1; break;
		case instruction::CONSULT_LIST: os << "CONSULT_LIST " << i.a1; break;
		case instruction::IS: os << "IS " << i.a1 << ", " << i.a2; break;
		case instruction::NUMEQUAL: os << "NUMEQUAL " << i.a1 << ", " << i.a2; break;
		case instruction::NUMUNEQUAL: os << "NUMUNEQUAL " << i.a1 << ", " << i.a2; break;
		case instruction::NUMLESS: os << "NUMLESS " << i.a1 << ", " << i.a2; break;
		case instruction::NUMLESSEQ: os << "NUMLESSEQ " << i.a1 << ", " << i.a2; break;
		case instruction::NUMGREATER: os << "numgreater " << i.a1 << ", " << i.a2; break;
		case instruction::NUMGREATEREQ: os << "numgreatereq " << i.a1 << ", " << i.a2; break;
		case instruction::UNIV: os << "UNIV " << i.a1 << ", " << i.a2; break;
		case instruction::FUNCTOR: os << "FUNCTOR " << i.a1 << ", " << i.a2 << ", " << i.a3; break;
		case instruction::ARG: os << "ARG " << i.a1 << ", " << i.a2 << ", " << i.a3; break;
		case instruction::NAME: os << "NAME " << i.a1 << ", " << i.a2; break;
		case instruction::MODEH: os << "MODEH " << i.a1 << ", " << i.a2; break;
		case instruction::MODEB: os << "MODEB " << i.a1 << ", " << i.a2; break;
		case instruction::MODEB3: os << "MODEB3 " << i.a1 << ", " << i.a2 << ", " << i.a3; break;
		case instruction::IMPLIES: os << "IMPLIES " << i.a1 << ", " << i.a2; break;
		case instruction::PREVENTS: os << "PREVENTS " << i.a1 << ", " << i.a2; break;
		case instruction::PREVENT: os << "PREVENT " << i.a1; break;
		case instruction::SET_PARAM: os << "SET " << i.a1 << ", " << i.a2; break;
		case instruction::GET_PARAM: os << "GET " << i.a1 << ", " << i.a2; break;
		case instruction::DETERMINATION: os << "DETERMINATION " << i.a1 << ", " << i.a2; break;
		case instruction::EXCLUDE_PRED: os << "EXCLUDE_PRED " << i.a1 << ", " << i.a2; break;
		case instruction::GENERALIZE0: os << "GENERALIZE0"; break;
		case instruction::GENERALIZE1: os << "GENERALIZE1 " << i.a1; break;
		default:
			std::cerr << "Error: unrecognized instruction: " << i.opcode << "\n";
			assert(false);
			std::exit(1);
		}
		return os;
	}

	Functor* term2functor(const term* t, std::map<const term*,id_type>& vmap)
	{
		if (is_cyclic(t)) throw cyclic_term();

		std::stack<std::pair<const term*,const term*>> pdl; // term and its parent
		std::map<const term*,Functor*> fmap;
		Functor* root = nullptr;

		try {
			pdl.push(std::make_pair(deref(t),nullptr));

			while (!pdl.empty()) {
				auto p = pdl.top();
				pdl.pop();
				t = p.first;
				const term* par = p.second;

				Functor* ptr = nullptr;
				// Handle node
				switch (t->get_type()) {
				case term::VAR: case term::EVAR:
					{
						assert(t == t->get_ptr());
						auto at = vmap.find(t);
						if (at == vmap.end()) {
							const auto newvar = Functor::unique_index();
							vmap.insert(std::make_pair(t,newvar));
							ptr = new Functor(newvar);
						} else {
							ptr = new Functor(at->second);
						}
						if (par) fmap.find(par)->second->steal(ptr);
						fmap.insert(std::make_pair(t,ptr));
						break;
					}
				case term::INT:
					{
						ptr = new Functor(Functor::make_int(t->get_int()));
						if (par) fmap.find(par)->second->steal(ptr);
						fmap.insert(std::make_pair(t,ptr));
						break;
					}
				case term::FLT:
					{
						ptr = new Functor(Functor::make_double(t->get_double()));
						if (par) fmap.find(par)->second->steal(ptr);
						fmap.insert(std::make_pair(t,ptr));
						break;
					}
				case term::CON:
					{
						ptr = new Functor(t->get_const());
						if (par) fmap.find(par)->second->steal(ptr);
						fmap.insert(std::make_pair(t,ptr));
						break;
					}
				case term::LIS:
				default:
					{
						ptr = (t->is_list() ? new Functor(pair_id) : new Functor(t->get_fun()));
						if (par) fmap.find(par)->second->steal(ptr);
						fmap.insert(std::make_pair(t,ptr));
						for (int k = t->arity(); k > 0; --k) {
							pdl.push(std::make_pair(deref(t+k),t));
						}
					}
				}

				if (!par) root = ptr;
			}
		} catch (...) {
			for (const auto& p : fmap) delete p.second;
			throw;
		}
		return root;
	}


}



